<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| contains the "web" middleware group. Now create something great!
|
*/

/*
|--------------------------------------------------------------------------
| Setup/Installation Routes (Must be first - no middleware)
| Accessible when application is not installed (checked by CheckInstallation middleware)
|--------------------------------------------------------------------------
*/
Route::prefix('setup')->name('setup.')->group(function () {
    Route::get('/', [App\Http\Controllers\SetupController::class, 'index'])->name('welcome');
    Route::get('/check-requirements', [App\Http\Controllers\SetupController::class, 'checkRequirements'])->name('check-requirements');
    Route::post('/test-database', [App\Http\Controllers\SetupController::class, 'testDatabase'])->name('test-database');
    Route::post('/install', [App\Http\Controllers\SetupController::class, 'install'])->name('install');
});

/*
|--------------------------------------------------------------------------
| Authentication Routes
|--------------------------------------------------------------------------
*/

// Root route - redirect to student login if not authenticated
Route::get('/', function () {
    if (Auth::check()) {
        $user = Auth::user();
        if ($user->role === 'admin') {
            return redirect()->route('admin.dashboard');
        } elseif ($user->role === 'instructor') {
            return redirect()->route('instructor.dashboard');
        } else {
            return redirect()->route('student.dashboard');
        }
    }
    return redirect()->route('student.login');
})->name('home');

// Student Login Routes - Use guest middleware to redirect if already logged in
Route::middleware(['guest', 'throttle:login'])->group(function () {
    Route::get('/login', [App\Http\Controllers\Auth\LoginController::class, 'showStudentLoginForm'])->name('student.login');
    Route::post('/login', [App\Http\Controllers\Auth\LoginController::class, 'login'])->name('login.post');
});

// Admin Login Routes - Use guest middleware to redirect if already logged in
Route::middleware(['guest', 'throttle:login'])->group(function () {
    Route::get('/admin', [App\Http\Controllers\Auth\LoginController::class, 'showAdminLoginForm'])->name('admin.login');
    Route::post('/admin', [App\Http\Controllers\Auth\LoginController::class, 'login'])->name('admin.login.post');
});

// Instructor Login Routes - Use guest middleware to redirect if already logged in
Route::middleware(['guest', 'throttle:login'])->group(function () {
    Route::get('/instructor', [App\Http\Controllers\Auth\LoginController::class, 'showInstructorLoginForm'])->name('instructor.login');
    Route::post('/instructor', [App\Http\Controllers\Auth\LoginController::class, 'login'])->name('instructor.login.post');
});

// Logout Route
Route::post('/logout', [App\Http\Controllers\Auth\LoginController::class, 'logout'])->name('logout');

// Password Reset Routes
Route::middleware('throttle:password-reset')->group(function () {
    Route::get('/password/reset', [App\Http\Controllers\Auth\ForgotPasswordController::class, 'showLinkRequestForm'])
        ->name('password.request');

    Route::post('/password/email', [App\Http\Controllers\Auth\ForgotPasswordController::class, 'sendResetLinkEmail'])
        ->name('password.email');
});

Route::get('/password/reset/{token}', [App\Http\Controllers\Auth\ResetPasswordController::class, 'showResetForm'])
    ->name('password.reset');

Route::post('/password/reset', [App\Http\Controllers\Auth\ResetPasswordController::class, 'reset'])
    ->name('password.update');

/*
|--------------------------------------------------------------------------
| Admin Routes
|--------------------------------------------------------------------------
*/
Route::prefix('admin')->name('admin.')->middleware(['auth', 'role:admin'])->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\AdminController::class, 'dashboard'])->name('dashboard');
    
    // Student Management Routes
    Route::resource('students', App\Http\Controllers\Admin\StudentController::class);
    Route::post('students/{student}/delivery-status', [App\Http\Controllers\Admin\StudentController::class, 'updateDeliveryStatus'])->name('students.delivery-status');
    
    // Instructor Management Routes
    Route::resource('instructors', App\Http\Controllers\Admin\InstructorController::class);
    
    // Admin Management Routes
    Route::resource('admins', App\Http\Controllers\Admin\AdminManagementController::class);
    
    // Course Management Routes
    Route::resource('courses', App\Http\Controllers\Admin\CourseController::class);
    
    // Course Chapter Management Routes (nested under courses)
    Route::get('courses/{course}/chapters/create', [App\Http\Controllers\Admin\CourseChapterController::class, 'create'])->name('courses.chapters.create');
    Route::post('courses/{course}/chapters', [App\Http\Controllers\Admin\CourseChapterController::class, 'store'])->name('courses.chapters.store');
    Route::get('courses/{course}/chapters/{chapter}', [App\Http\Controllers\Admin\CourseChapterController::class, 'show'])->name('courses.chapters.show');
    Route::get('courses/{course}/chapters/{chapter}/edit', [App\Http\Controllers\Admin\CourseChapterController::class, 'edit'])->name('courses.chapters.edit');
    Route::put('courses/{course}/chapters/{chapter}', [App\Http\Controllers\Admin\CourseChapterController::class, 'update'])->name('courses.chapters.update');
    Route::delete('courses/{course}/chapters/{chapter}', [App\Http\Controllers\Admin\CourseChapterController::class, 'destroy'])->name('courses.chapters.destroy');

    // Course Offerings Management Routes
    Route::resource('course-offerings', App\Http\Controllers\Admin\CourseOfferingController::class);

    // Enrollments Management Routes
    Route::resource('enrollments', App\Http\Controllers\Admin\EnrollmentController::class);

    // Class Videos Management Routes
    Route::resource('class-videos', App\Http\Controllers\Admin\ClassVideoController::class);

    // Certificates Management Routes
    Route::resource('certificates', App\Http\Controllers\Admin\CertificateController::class);
    
    // Resources Management Routes
    Route::resource('resources', App\Http\Controllers\Admin\ResourceController::class);
    Route::get('resources/{resource}/download', [App\Http\Controllers\Admin\ResourceController::class, 'download'])->name('resources.download');

    // Student Finances Management Routes
    Route::get('student-finances', [App\Http\Controllers\Admin\StudentFinanceController::class, 'index'])->name('student-finances.index');

    // Fee Payments Routes
    Route::get('fee-payments', [App\Http\Controllers\Admin\FeePaymentController::class, 'index'])->name('fee-payments.index');
    Route::post('fee-payments', [App\Http\Controllers\Admin\FeePaymentController::class, 'store'])->name('fee-payments.store');
    Route::get('fee-payments/{payment}/edit', [App\Http\Controllers\Admin\FeePaymentController::class, 'edit'])->name('fee-payments.edit');
    Route::match(['put', 'post'], 'fee-payments/{payment}', [App\Http\Controllers\Admin\FeePaymentController::class, 'update'])->name('fee-payments.update');
    Route::delete('fee-payments/{payment}', [App\Http\Controllers\Admin\FeePaymentController::class, 'destroy'])->name('fee-payments.destroy');
    
    // Profile Routes
    Route::get('/profile', [App\Http\Controllers\Admin\ProfileController::class, 'index'])->name('profile.index');
    Route::match(['put', 'post'], '/profile', [App\Http\Controllers\Admin\ProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [App\Http\Controllers\Admin\ProfileController::class, 'updatePassword'])->name('profile.updatePassword');
    
    // Announcements Routes
    Route::get('/announcements', [App\Http\Controllers\Admin\AnnouncementController::class, 'index'])->name('announcements.index');
    Route::post('/announcements', [App\Http\Controllers\Admin\AnnouncementController::class, 'store'])->name('announcements.store');
    
    // Settings Routes
    Route::get('/settings', [App\Http\Controllers\Admin\SettingsController::class, 'index'])->name('settings.index');
    Route::put('/settings', [App\Http\Controllers\Admin\SettingsController::class, 'update'])->name('settings.update');
    Route::post('/settings/test-email', [App\Http\Controllers\Admin\SettingsController::class, 'testEmail'])->name('settings.test-email');
    
    // Quiz Management Routes
    Route::resource('quizzes', App\Http\Controllers\Admin\QuizController::class);
    Route::post('quizzes/{quiz}/questions', [App\Http\Controllers\Admin\QuizController::class, 'storeQuestion'])->name('quizzes.questions.store');
    Route::put('quizzes/{quiz}/questions/{question}', [App\Http\Controllers\Admin\QuizController::class, 'updateQuestion'])->name('quizzes.questions.update');
    Route::delete('quizzes/{quiz}/questions/{question}', [App\Http\Controllers\Admin\QuizController::class, 'deleteQuestion'])->name('quizzes.questions.destroy');
    Route::get('quizzes/{quiz}/export', [App\Http\Controllers\Admin\QuizController::class, 'export'])->name('quizzes.export');
    Route::get('quizzes/import', [App\Http\Controllers\Admin\QuizController::class, 'showImport'])->name('quizzes.import');
    Route::post('quizzes/import', [App\Http\Controllers\Admin\QuizController::class, 'import'])->name('quizzes.import.store');
    Route::get('quizzes/template/download', [App\Http\Controllers\Admin\QuizController::class, 'downloadTemplate'])->name('quizzes.template.download');
});

/*
|--------------------------------------------------------------------------
| Instructor Routes
|--------------------------------------------------------------------------
*/
Route::prefix('instructor')->name('instructor.')->middleware(['auth', 'role:instructor'])->group(function () {
    // Resources
    Route::get('/resources', [App\Http\Controllers\Instructor\ResourceController::class, 'index'])->name('resources.index');
    Route::get('/resources/{resource}', [App\Http\Controllers\Instructor\ResourceController::class, 'show'])->name('resources.show');
    Route::get('/resources/{resource}/download', [App\Http\Controllers\Instructor\ResourceController::class, 'download'])->name('resources.download');
    // Dashboard
    Route::get('/dashboard', [App\Http\Controllers\InstructorController::class, 'dashboard'])->name('dashboard');
    
    // My Courses
    Route::get('/courses', [App\Http\Controllers\Instructor\CourseController::class, 'index'])->name('courses.index');
    Route::get('/courses/{course}', [App\Http\Controllers\Instructor\CourseController::class, 'show'])->name('courses.show');
    
    // My Students
    Route::get('/students', [App\Http\Controllers\Instructor\StudentController::class, 'index'])->name('students.index');
    Route::get('/students/{student}', [App\Http\Controllers\Instructor\StudentController::class, 'show'])->name('students.show');
    
    // Class Videos
    Route::get('/videos', [App\Http\Controllers\Instructor\VideoController::class, 'index'])->name('videos.index');
    Route::get('/videos/create', [App\Http\Controllers\Instructor\VideoController::class, 'create'])->name('videos.create');
    Route::post('/videos', [App\Http\Controllers\Instructor\VideoController::class, 'store'])->name('videos.store');
    Route::get('/videos/{video}', [App\Http\Controllers\Instructor\VideoController::class, 'show'])->name('videos.show');
    Route::get('/videos/{video}/edit', [App\Http\Controllers\Instructor\VideoController::class, 'edit'])->name('videos.edit');
    Route::put('/videos/{video}', [App\Http\Controllers\Instructor\VideoController::class, 'update'])->name('videos.update');
    Route::delete('/videos/{video}', [App\Http\Controllers\Instructor\VideoController::class, 'destroy'])->name('videos.destroy');
    
    // Assignments
    Route::resource('assignments', App\Http\Controllers\Instructor\AssignmentController::class);
    Route::get('/assignments/{id}/submissions', [App\Http\Controllers\Instructor\AssignmentController::class, 'submissions'])->name('assignments.submissions');
    Route::post('/assignments/submissions/{id}/grade', [App\Http\Controllers\Instructor\AssignmentController::class, 'grade'])->name('assignments.grade');
    
    // Profile Routes
    Route::get('/profile', [App\Http\Controllers\Instructor\ProfileController::class, 'index'])->name('profile.index');
    Route::match(['put', 'post'], '/profile', [App\Http\Controllers\Instructor\ProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [App\Http\Controllers\Instructor\ProfileController::class, 'updatePassword'])->name('profile.updatePassword');
    
    // Quiz Management Routes (instructors use admin controller but with instructor routes)
    Route::resource('quizzes', App\Http\Controllers\Admin\QuizController::class);
    Route::post('quizzes/{quiz}/questions', [App\Http\Controllers\Admin\QuizController::class, 'storeQuestion'])->name('quizzes.questions.store');
    Route::put('quizzes/{quiz}/questions/{question}', [App\Http\Controllers\Admin\QuizController::class, 'updateQuestion'])->name('quizzes.questions.update');
    Route::delete('quizzes/{quiz}/questions/{question}', [App\Http\Controllers\Admin\QuizController::class, 'deleteQuestion'])->name('quizzes.questions.destroy');
    Route::get('quizzes/{quiz}/export', [App\Http\Controllers\Admin\QuizController::class, 'export'])->name('quizzes.export');
    Route::get('quizzes/import', [App\Http\Controllers\Admin\QuizController::class, 'showImport'])->name('quizzes.import');
    Route::post('quizzes/import', [App\Http\Controllers\Admin\QuizController::class, 'import'])->name('quizzes.import.store');
    Route::get('quizzes/template/download', [App\Http\Controllers\Admin\QuizController::class, 'downloadTemplate'])->name('quizzes.template.download');
});

/*
|--------------------------------------------------------------------------
| Student Routes
|--------------------------------------------------------------------------
*/
Route::prefix('student')->name('student.')->middleware(['auth', 'role:student'])->group(function () {
    // Dashboard
    Route::get('/dashboard', [App\Http\Controllers\StudentController::class, 'dashboard'])->name('dashboard');
    
    // My Courses
    Route::get('/courses', [App\Http\Controllers\Student\CourseController::class, 'index'])->name('courses.index');
    Route::get('/courses/{course}', [App\Http\Controllers\Student\CourseController::class, 'show'])->name('courses.show');
    
    // Assignments
    Route::get('/assignments', [App\Http\Controllers\Student\AssignmentController::class, 'index'])->name('assignments.index');
    Route::get('/assignments/{id}', [App\Http\Controllers\Student\AssignmentController::class, 'show'])->name('assignments.show');
    Route::post('/assignments/{id}/submit', [App\Http\Controllers\Student\AssignmentController::class, 'submit'])->name('assignments.submit');
    
    // Class Videos
    Route::get('/videos', [App\Http\Controllers\Student\VideoController::class, 'index'])->name('videos.index');
    Route::get('/videos/{id}', [App\Http\Controllers\Student\VideoController::class, 'show'])->name('videos.show');
    
    // Certificates
    Route::get('/certificates', [App\Http\Controllers\Student\CertificateController::class, 'index'])->name('certificates.index');
    Route::get('/certificates/{certificate}', [App\Http\Controllers\Student\CertificateController::class, 'show'])->name('certificates.show');
    
    // Resources
    Route::get('/resources', [App\Http\Controllers\Student\ResourceController::class, 'index'])->name('resources.index');
    Route::get('/resources/{resource}', [App\Http\Controllers\Student\ResourceController::class, 'show'])->name('resources.show');
    Route::get('/resources/{resource}/download', [App\Http\Controllers\Student\ResourceController::class, 'download'])->name('resources.download');
    
    // Profile Routes
    Route::get('/profile', [App\Http\Controllers\Student\ProfileController::class, 'index'])->name('profile.index');
    Route::match(['put', 'post'], '/profile', [App\Http\Controllers\Student\ProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [App\Http\Controllers\Student\ProfileController::class, 'updatePassword'])->name('profile.updatePassword');
    Route::post('/profile/notify-payment', [App\Http\Controllers\Student\ProfileController::class, 'notifyPayment'])->name('profile.notifyPayment');
    
    // Books Routes
    Route::get('/books', [App\Http\Controllers\Student\BookController::class, 'index'])->name('books.index');
    Route::get('/books/{book}', [App\Http\Controllers\Student\BookController::class, 'show'])->name('books.show');
    Route::get('/books/{book}/chapters/{chapter}', [App\Http\Controllers\Student\BookController::class, 'readChapter'])->name('books.chapters.read');
    Route::get('/books/{book}/chapters/{chapter}/practice', [App\Http\Controllers\Student\BookController::class, 'showPractice'])->name('books.chapters.practice');
    Route::get('/books/{book}/chapters/{chapter}/download', [App\Http\Controllers\Student\BookController::class, 'downloadChapter'])->name('books.chapters.download');
    
    // Book Notes Routes (AJAX)
    Route::post('/books/{book}/chapters/{chapter}/notes', [App\Http\Controllers\Student\BookController::class, 'saveNote'])->name('books.notes.store');
    Route::put('/books/notes/{note}', [App\Http\Controllers\Student\BookController::class, 'updateNote'])->name('books.notes.update');
    Route::delete('/books/notes/{note}', [App\Http\Controllers\Student\BookController::class, 'deleteNote'])->name('books.notes.destroy');
    Route::get('/books/{book}/notes', [App\Http\Controllers\Student\BookController::class, 'getNotes'])->name('books.notes.index');
    
    // Book Progress Routes (AJAX)
    Route::post('/books/{book}/chapters/{chapter}/progress', [App\Http\Controllers\Student\BookController::class, 'saveProgress'])->name('books.progress.store');
    
    // Bookmark Routes (AJAX)
    Route::post('/books/{book}/chapters/{chapter}/bookmarks', [App\Http\Controllers\Student\BookController::class, 'createBookmark'])->name('books.bookmarks.store');
    Route::delete('/books/bookmarks/{bookmark}', [App\Http\Controllers\Student\BookController::class, 'deleteBookmark'])->name('books.bookmarks.destroy');
    
    // Quiz Routes
    Route::get('/quizzes', [App\Http\Controllers\Student\QuizController::class, 'index'])->name('quizzes.index');
    Route::get('/quizzes/history', [App\Http\Controllers\Student\QuizController::class, 'history'])->name('quizzes.history');
    Route::get('/quizzes/analytics', [App\Http\Controllers\Student\QuizController::class, 'analytics'])->name('quizzes.analytics');
    Route::get('/quizzes/bookmarks', [App\Http\Controllers\Student\QuizController::class, 'bookmarks'])->name('quizzes.bookmarks');
    Route::get('/quizzes/review-incorrect', [App\Http\Controllers\Student\QuizController::class, 'reviewIncorrect'])->name('quizzes.review-incorrect');
    Route::get('/quizzes/questions/{question}/retake', [App\Http\Controllers\Student\QuizController::class, 'retakeQuestion'])->name('quizzes.retake-question');
    Route::post('/quizzes/questions/{question}/bookmark', [App\Http\Controllers\Student\QuizController::class, 'toggleBookmark'])->name('quizzes.toggle-bookmark');
    Route::put('/quizzes/bookmarks/{bookmark}/notes', [App\Http\Controllers\Student\QuizController::class, 'updateBookmarkNotes'])->name('quizzes.update-bookmark-notes');
    Route::get('/quizzes/{quiz}', [App\Http\Controllers\Student\QuizController::class, 'show'])->name('quizzes.show');
    Route::post('/quizzes/{quiz}/submit-answer', [App\Http\Controllers\Student\QuizController::class, 'submitAnswer'])->name('quizzes.submit-answer');
    Route::post('/quizzes/{quiz}/complete', [App\Http\Controllers\Student\QuizController::class, 'complete'])->name('quizzes.complete');
    Route::get('/quizzes/{quiz}/result/{attempt}', [App\Http\Controllers\Student\QuizController::class, 'result'])->name('quizzes.result');
});

/*
|--------------------------------------------------------------------------
| Notification Routes (All Roles)
|--------------------------------------------------------------------------
*/
Route::middleware('auth')->group(function () {
    Route::get('/notifications', [App\Http\Controllers\NotificationController::class, 'index'])->name('notifications.index');
    Route::get('/notifications/{notification}', [App\Http\Controllers\NotificationController::class, 'show'])->name('notifications.show');
    Route::post('/notifications/{notification}/read', [App\Http\Controllers\NotificationController::class, 'markAsRead'])->name('notifications.mark-read');
    Route::post('/notifications/mark-all-read', [App\Http\Controllers\NotificationController::class, 'markAllAsRead'])->name('notifications.mark-all-read');
    Route::delete('/notifications/{notification}', [App\Http\Controllers\NotificationController::class, 'destroy'])->name('notifications.destroy');
    
    // AJAX Routes for notification bell
    Route::get('/api/notifications/unread-count', [App\Http\Controllers\NotificationController::class, 'unreadCount'])->name('notifications.unread-count');
    Route::get('/api/notifications/recent', [App\Http\Controllers\NotificationController::class, 'recent'])->name('notifications.recent');
});
