<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\V1\AuthController;

/*
|--------------------------------------------------------------------------
| API v1 Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for version 1 of the API.
| These routes are loaded by the RouteServiceProvider.
|
*/

// Routes are already under /api/v1 prefix from routes/api.php
// No need to add another /v1 prefix here
Route::group([], function () {
    
    // Public routes (no authentication required) - with rate limiting
    Route::middleware('throttle:api-auth')->group(function () {
        Route::post('/auth/login', [AuthController::class, 'login']);
        Route::post('/auth/password/email', [AuthController::class, 'sendPasswordResetLink']);
        Route::post('/auth/password/reset', [AuthController::class, 'resetPassword']);
    });
    
    // Protected routes (authentication required) - with rate limiting and input sanitization
    Route::middleware(['auth:api', 'throttle:api-endpoints', 'sanitize'])->group(function () {
        
        // Authentication routes
        Route::post('/auth/logout', [AuthController::class, 'logout']);
        Route::get('/auth/user', [AuthController::class, 'user']);
        
        // Student routes
        Route::prefix('student')->middleware('api.role:student')->group(function () {
            // Dashboard
            Route::get('/dashboard', [\App\Http\Controllers\Api\V1\Student\DashboardController::class, 'index']);
            
            // Courses
            Route::get('/courses', [\App\Http\Controllers\Api\V1\Student\CourseController::class, 'index']);
            Route::get('/courses/{id}', [\App\Http\Controllers\Api\V1\Student\CourseController::class, 'show']);
            
            // Assignments
            Route::get('/assignments', [\App\Http\Controllers\Api\V1\Student\AssignmentController::class, 'index']);
            Route::get('/assignments/{id}', [\App\Http\Controllers\Api\V1\Student\AssignmentController::class, 'show']);
            Route::post('/assignments/{id}/submit', [\App\Http\Controllers\Api\V1\Student\AssignmentController::class, 'submit']);
            
            // Books
            Route::get('/books', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'index']);
            Route::get('/books/{id}', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'show']);
            Route::get('/books/{bookId}/chapters/{chapterId}', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'getChapter']);
            Route::get('/books/{bookId}/chapters/{chapterId}/practices', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'getPractices']);
            Route::get('/books/{bookId}/chapters/{chapterId}/practices/{practiceId}', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'getPractice']);
            Route::post('/books/{bookId}/chapters/{chapterId}/progress', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'saveProgress']);
            Route::get('/books/{bookId}/chapters/{chapterId}/notes', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'getNotes']);
            Route::post('/books/{bookId}/chapters/{chapterId}/notes', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'saveNote']);
            Route::delete('/books/{bookId}/chapters/{chapterId}/notes/{noteId}', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'deleteNote']);
            Route::post('/books/{bookId}/chapters/{chapterId}/bookmarks', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'createBookmark']);
            Route::delete('/books/{bookId}/chapters/{chapterId}/bookmarks', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'deleteBookmark']);
            Route::get('/books/{bookId}/chapters/{chapterId}/download', [\App\Http\Controllers\Api\V1\Student\BookController::class, 'downloadChapter']);
            
            // Quizzes
            Route::get('/quizzes', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'index']);
            Route::get('/quizzes/{id}', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'show']);
            Route::post('/quizzes/{id}/start', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'start']);
            Route::post('/quizzes/{id}/answer', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'submitAnswer']);
            Route::post('/quizzes/{id}/complete', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'complete']);
            Route::get('/quizzes/{id}/results/{attemptId}', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'result']);
            Route::get('/quizzes/history', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'history']);
            Route::get('/quizzes/analytics', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'analytics']);
            Route::get('/quizzes/review-incorrect', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'reviewIncorrect']);
            Route::post('/quizzes/questions/{questionId}/bookmark', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'toggleBookmark']);
            Route::get('/quizzes/bookmarks', [\App\Http\Controllers\Api\V1\Student\QuizController::class, 'bookmarks']);
            
            // Videos
            Route::get('/videos', [\App\Http\Controllers\Api\V1\Student\VideoController::class, 'index']);
            Route::get('/videos/{id}', [\App\Http\Controllers\Api\V1\Student\VideoController::class, 'show']);
            
            // Certificates
            Route::get('/certificates', [\App\Http\Controllers\Api\V1\Student\CertificateController::class, 'index']);
            Route::get('/certificates/{id}', [\App\Http\Controllers\Api\V1\Student\CertificateController::class, 'show']);
            
            // Profile
            Route::get('/profile', [\App\Http\Controllers\Api\V1\Student\ProfileController::class, 'index']);
            Route::put('/profile', [\App\Http\Controllers\Api\V1\Student\ProfileController::class, 'update']);
            Route::post('/profile/password', [\App\Http\Controllers\Api\V1\Student\ProfileController::class, 'changePassword']);
        });
        
        // Instructor routes
        Route::prefix('instructor')->middleware('api.role:instructor')->group(function () {
            // Dashboard
            Route::get('/dashboard', [\App\Http\Controllers\Api\V1\Instructor\DashboardController::class, 'index']);
            
            // Courses
            Route::get('/courses', [\App\Http\Controllers\Api\V1\Instructor\CourseController::class, 'index']);
            Route::get('/courses/{id}', [\App\Http\Controllers\Api\V1\Instructor\CourseController::class, 'show']);
            
            // Students
            Route::get('/students', [\App\Http\Controllers\Api\V1\Instructor\StudentController::class, 'index']);
            Route::get('/students/{id}', [\App\Http\Controllers\Api\V1\Instructor\StudentController::class, 'show']);
            
            // Assignments
            Route::get('/assignments', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'index']);
            Route::post('/assignments', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'store']);
            Route::get('/assignments/{id}', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'show']);
            Route::put('/assignments/{id}', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'update']);
            Route::delete('/assignments/{id}', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'destroy']);
            Route::get('/assignments/{id}/submissions', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'submissions']);
            Route::post('/assignments/{assignmentId}/submissions/{submissionId}/grade', [\App\Http\Controllers\Api\V1\Instructor\AssignmentController::class, 'grade']);
            
            // Videos
            Route::get('/videos', [\App\Http\Controllers\Api\V1\Instructor\VideoController::class, 'index']);
            Route::post('/videos', [\App\Http\Controllers\Api\V1\Instructor\VideoController::class, 'store']);
            Route::get('/videos/{id}', [\App\Http\Controllers\Api\V1\Instructor\VideoController::class, 'show']);
            Route::put('/videos/{id}', [\App\Http\Controllers\Api\V1\Instructor\VideoController::class, 'update']);
            Route::delete('/videos/{id}', [\App\Http\Controllers\Api\V1\Instructor\VideoController::class, 'destroy']);
            
            // Profile
            Route::get('/profile', [\App\Http\Controllers\Api\V1\Instructor\ProfileController::class, 'index']);
            Route::put('/profile', [\App\Http\Controllers\Api\V1\Instructor\ProfileController::class, 'update']);
            Route::post('/profile/password', [\App\Http\Controllers\Api\V1\Instructor\ProfileController::class, 'changePassword']);
        });
        
        // Admin routes
        Route::prefix('admin')->middleware('api.role:admin')->group(function () {
            // Dashboard
            Route::get('/dashboard', [\App\Http\Controllers\Api\V1\Admin\DashboardController::class, 'index']);
            
            // Students
            Route::get('/students', [\App\Http\Controllers\Api\V1\Admin\StudentController::class, 'index']);
            Route::post('/students', [\App\Http\Controllers\Api\V1\Admin\StudentController::class, 'store']);
            Route::get('/students/{id}', [\App\Http\Controllers\Api\V1\Admin\StudentController::class, 'show']);
            Route::put('/students/{id}', [\App\Http\Controllers\Api\V1\Admin\StudentController::class, 'update']);
            Route::delete('/students/{id}', [\App\Http\Controllers\Api\V1\Admin\StudentController::class, 'destroy']);
            Route::post('/students/{id}/delivery-status', [\App\Http\Controllers\Api\V1\Admin\StudentController::class, 'updateDeliveryStatus']);
            
            // Instructors
            Route::get('/instructors', [\App\Http\Controllers\Api\V1\Admin\InstructorController::class, 'index']);
            Route::post('/instructors', [\App\Http\Controllers\Api\V1\Admin\InstructorController::class, 'store']);
            Route::get('/instructors/{id}', [\App\Http\Controllers\Api\V1\Admin\InstructorController::class, 'show']);
            Route::put('/instructors/{id}', [\App\Http\Controllers\Api\V1\Admin\InstructorController::class, 'update']);
            Route::delete('/instructors/{id}', [\App\Http\Controllers\Api\V1\Admin\InstructorController::class, 'destroy']);
            Route::post('/instructors/bulk-action', [\App\Http\Controllers\Api\V1\Admin\InstructorController::class, 'bulkAction']);
            
            // Fee Payments
            Route::get('/fee-payments', [\App\Http\Controllers\Api\V1\Admin\FeePaymentController::class, 'index']);
            Route::post('/fee-payments', [\App\Http\Controllers\Api\V1\Admin\FeePaymentController::class, 'store']);
            Route::put('/fee-payments/{id}', [\App\Http\Controllers\Api\V1\Admin\FeePaymentController::class, 'update']);
            Route::delete('/fee-payments/{id}', [\App\Http\Controllers\Api\V1\Admin\FeePaymentController::class, 'destroy']);
            
            // Profile
            Route::get('/profile', [\App\Http\Controllers\Api\V1\Admin\ProfileController::class, 'index']);
            Route::put('/profile', [\App\Http\Controllers\Api\V1\Admin\ProfileController::class, 'update']);
            Route::post('/profile/password', [\App\Http\Controllers\Api\V1\Admin\ProfileController::class, 'changePassword']);
        });
        
        // Common routes (available to all authenticated users)
        // Notifications
        Route::get('/notifications', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'index']);
        Route::get('/notifications/unread-count', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'unreadCount']);
        Route::get('/notifications/recent', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'recent']);
        Route::get('/notifications/{id}', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'show']);
        Route::post('/notifications/{id}/read', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'markAsRead']);
        Route::post('/notifications/mark-all-read', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'markAllAsRead']);
        Route::delete('/notifications/{id}', [\App\Http\Controllers\Api\V1\Student\NotificationController::class, 'destroy']);
        
        // Device Tokens (for push notifications)
        Route::post('/device-token/register', [\App\Http\Controllers\Api\V1\Student\DeviceTokenController::class, 'register']);
        Route::post('/device-token/unregister', [\App\Http\Controllers\Api\V1\Student\DeviceTokenController::class, 'unregister']);
    });
});
