@extends('layouts.student')

@section('title', 'Performance Analytics')
@section('page-title', 'Performance Analytics')

@push('styles')
<style>
    .analytics-container {
        max-width: 1400px;
        margin: 0 auto;
    }
    
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }
    
    .stat-card {
        background: #2d2d2d;
        padding: 25px;
        border-radius: 8px;
        border: 1px solid #404040;
        text-align: center;
    }
    
    .stat-card-icon {
        font-size: 36px;
        margin-bottom: 15px;
        color: #4a9eff;
    }
    
    .stat-card-value {
        font-size: 32px;
        font-weight: bold;
        color: #e0e0e0;
        margin-bottom: 8px;
    }
    
    .stat-card-label {
        color: #888;
        font-size: 14px;
    }
    
    .chart-section {
        background: #2d2d2d;
        padding: 30px;
        border-radius: 8px;
        margin-bottom: 30px;
        border: 1px solid #404040;
    }
    
    .chart-section h3 {
        color: #4a9eff;
        margin-bottom: 20px;
    }
    
    .chart-container {
        height: 300px;
        position: relative;
    }
    
    .performance-table {
        width: 100%;
        border-collapse: collapse;
    }
    
    .performance-table th {
        background: #1a1a1a;
        padding: 15px;
        text-align: left;
        color: #4a9eff;
        border-bottom: 2px solid #404040;
    }
    
    .performance-table td {
        padding: 15px;
        border-bottom: 1px solid #404040;
        color: #e0e0e0;
    }
    
    .performance-table tr:hover {
        background: #1a1a1a;
    }
    
    .score-badge {
        padding: 5px 12px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: bold;
    }
    
    .score-excellent {
        background: #1a3a1a;
        color: #6bff6b;
    }
    
    .score-good {
        background: #3a3a1a;
        color: #ffa500;
    }
    
    .score-poor {
        background: #3a1a1a;
        color: #ff6b6b;
    }
    
    .level-badge {
        display: inline-block;
        padding: 4px 10px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: bold;
    }
    
    .level-beginner {
        background: #1a3a1a;
        color: #6bff6b;
    }
    
    .level-intermediate {
        background: #3a3a1a;
        color: #ffa500;
    }
    
    .level-advanced {
        background: #3a1a1a;
        color: #ff6b6b;
    }
    
    .weak-area-card {
        background: #1a1a1a;
        padding: 20px;
        border-radius: 6px;
        border-left: 4px solid #ff6b6b;
        margin-bottom: 15px;
    }
    
    .weak-area-card h4 {
        color: #e0e0e0;
        margin-bottom: 10px;
    }
    
    .weak-area-card p {
        color: #888;
        font-size: 14px;
        margin: 0;
    }
</style>
@endpush

@section('content')
    <div class="analytics-container">
        <!-- Overall Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-card-icon">
                    <i class="fas fa-tasks"></i>
                </div>
                <div class="stat-card-value">{{ $totalAttempts }}</div>
                <div class="stat-card-label">Total Attempts</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-card-icon">
                    <i class="fas fa-question-circle"></i>
                </div>
                <div class="stat-card-value">{{ $totalQuizzes }}</div>
                <div class="stat-card-label">Quizzes Completed</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-card-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="stat-card-value">{{ number_format($averageScore, 1) }}%</div>
                <div class="stat-card-label">Average Score</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-card-icon">
                    <i class="fas fa-bullseye"></i>
                </div>
                <div class="stat-card-value">{{ number_format($overallAccuracy, 1) }}%</div>
                <div class="stat-card-label">Overall Accuracy</div>
            </div>
        </div>
        
        <!-- Performance Over Time Chart -->
        <div class="chart-section">
            <h3><i class="fas fa-chart-area"></i> Performance Over Time (Last 30 Days)</h3>
            <div class="chart-container">
                <canvas id="performanceChart"></canvas>
            </div>
        </div>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin-bottom: 30px;">
            <!-- Performance by Level -->
            <div class="chart-section">
                <h3><i class="fas fa-layer-group"></i> Performance by Level</h3>
                <div class="chart-container">
                    <canvas id="levelChart"></canvas>
                </div>
            </div>
            
            <!-- Weak Areas -->
            <div class="chart-section">
                <h3><i class="fas fa-exclamation-triangle"></i> Areas Needing Improvement</h3>
                @if($weakAreas->count() > 0)
                    @foreach($weakAreas as $weakArea)
                        <div class="weak-area-card">
                            <h4>{{ $weakArea['quiz']->title }}</h4>
                            <p>Average Score: <strong style="color: #ff6b6b;">{{ number_format($weakArea['avg_score'], 1) }}%</strong></p>
                            <a href="{{ route('student.quizzes.show', $weakArea['quiz']->id) }}" class="btn btn-primary" style="margin-top: 10px; padding: 8px 16px; font-size: 13px;">
                                <i class="fas fa-redo"></i> Retake Quiz
                            </a>
                        </div>
                    @endforeach
                @else
                    <p style="color: #888; text-align: center; padding: 40px;">Great job! No weak areas identified.</p>
                @endif
            </div>
        </div>
        
        <!-- Top Performing Quizzes -->
        <div class="chart-section">
            <h3><i class="fas fa-trophy"></i> Top Performing Quizzes</h3>
            <table class="performance-table">
                <thead>
                    <tr>
                        <th>Quiz</th>
                        <th>Level</th>
                        <th>Best Score</th>
                        <th>Average Score</th>
                        <th>Attempts</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($quizPerformance as $performance)
                        @php
                            $scoreClass = $performance['best_score'] >= 80 ? 'score-excellent' : ($performance['best_score'] >= 60 ? 'score-good' : 'score-poor');
                            $levelClass = 'level-' . strtolower($performance['quiz']->level_name);
                        @endphp
                        <tr>
                            <td><strong>{{ $performance['quiz']->title }}</strong></td>
                            <td>
                                <span class="level-badge {{ $levelClass }}">
                                    {{ $performance['quiz']->level_name }}
                                </span>
                            </td>
                            <td>
                                <span class="score-badge {{ $scoreClass }}">
                                    {{ number_format($performance['best_score'], 1) }}%
                                </span>
                            </td>
                            <td>{{ number_format($performance['avg_score'], 1) }}%</td>
                            <td>{{ $performance['attempts'] }}</td>
                            <td>
                                <a href="{{ route('student.quizzes.show', $performance['quiz']->id) }}" class="btn btn-secondary" style="padding: 6px 12px; font-size: 12px;">
                                    <i class="fas fa-redo"></i> Retake
                                </a>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" style="text-align: center; color: #888; padding: 40px;">
                                No quiz attempts yet. Start taking quizzes to see your performance!
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script>
    // Performance Over Time Chart
    const performanceCtx = document.getElementById('performanceChart');
    if (performanceCtx) {
        const performanceData = @json($performanceData);
        
        new Chart(performanceCtx, {
            type: 'line',
            data: {
                labels: performanceData.map(d => d.date),
                datasets: [{
                    label: 'Score (%)',
                    data: performanceData.map(d => d.score),
                    borderColor: '#4a9eff',
                    backgroundColor: 'rgba(74, 158, 255, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        ticks: {
                            color: '#888'
                        },
                        grid: {
                            color: '#404040'
                        }
                    },
                    x: {
                        ticks: {
                            color: '#888'
                        },
                        grid: {
                            color: '#404040'
                        }
                    }
                }
            }
        });
    }
    
    // Performance by Level Chart
    const levelCtx = document.getElementById('levelChart');
    if (levelCtx) {
        const levelData = @json($levelPerformance);
        const labels = levelData.map(l => l.level_name);
        const scores = levelData.map(l => l.avg_score);
        
        new Chart(levelCtx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Average Score (%)',
                    data: scores,
                    backgroundColor: [
                        'rgba(107, 255, 107, 0.6)',
                        'rgba(255, 165, 0, 0.6)',
                        'rgba(255, 107, 107, 0.6)'
                    ],
                    borderColor: [
                        '#6bff6b',
                        '#ffa500',
                        '#ff6b6b'
                    ],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        ticks: {
                            color: '#888'
                        },
                        grid: {
                            color: '#404040'
                        }
                    },
                    x: {
                        ticks: {
                            color: '#888'
                        },
                        grid: {
                            color: '#404040'
                        }
                    }
                }
            }
        });
    }
</script>
@endpush

