<?php

/**
 * Populate Flutter & Dart Course Curriculum
 * 
 * This script reads all chapter HTML files from the "Flutter Online Book Version" folder
 * and creates course chapters in the database for the Flutter & Dart course.
 * 
 * Usage: php populate_flutter_course_curriculum.php
 * 
 * Make sure to run this from the project root directory or adjust the paths accordingly.
 */

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\Course;
use App\Models\CourseChapter;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;

echo "========================================\n";
echo "Flutter Course Curriculum Populator\n";
echo "========================================\n\n";

// Step 1: Find the Flutter & Dart course
echo "Step 1: Finding Flutter & Dart course...\n";
$course = Course::where(function($query) {
    $query->where('name', 'like', '%Flutter%')
          ->orWhere('name', 'like', '%Dart%')
          ->orWhere('code', 'like', '%FLUTTER%')
          ->orWhere('code', 'like', '%DART%');
})->first();

if (!$course) {
    echo "❌ ERROR: Flutter & Dart course not found!\n";
    echo "Please create the course first or check the course name/code.\n";
    echo "\nLooking for courses with 'Flutter' or 'Dart' in name or code...\n";
    $courses = Course::where('name', 'like', '%Flutter%')
        ->orWhere('name', 'like', '%Dart%')
        ->orWhere('code', 'like', '%FLUTTER%')
        ->orWhere('code', 'like', '%DART%')
        ->get(['id', 'name', 'code']);
    
    if ($courses->count() > 0) {
        echo "Found courses:\n";
        foreach ($courses as $c) {
            echo "  - ID: {$c->id}, Name: {$c->name}, Code: {$c->code}\n";
        }
    } else {
        echo "No matching courses found.\n";
    }
    exit(1);
}

echo "✓ Found course: {$course->name} (ID: {$course->id}, Code: {$course->code})\n\n";

// Step 2: Check if chapters already exist
$existingChapters = CourseChapter::where('course_id', $course->id)->count();
if ($existingChapters > 0) {
    echo "⚠ WARNING: Course already has {$existingChapters} chapters.\n";
    
    // Check if running in non-interactive mode (cPanel terminal)
    $nonInteractive = isset($argv[1]) && $argv[1] === '--skip-existing';
    $deleteExisting = isset($argv[1]) && $argv[1] === '--delete-existing';
    
    if ($deleteExisting) {
        echo "Deleting existing chapters (--delete-existing flag)...\n";
        CourseChapter::where('course_id', $course->id)->delete();
        echo "✓ Deleted {$existingChapters} existing chapters.\n\n";
    } elseif ($nonInteractive) {
        echo "Skipping existing chapters (--skip-existing flag)...\n\n";
    } else {
        // Interactive mode
        echo "Do you want to:\n";
        echo "  1. Delete existing chapters and create new ones\n";
        echo "  2. Skip existing chapters and only add missing ones\n";
        echo "  3. Cancel\n";
        echo "Enter choice (1/2/3): ";
        $handle = fopen("php://stdin", "r");
        $choice = trim(fgets($handle));
        fclose($handle);
        
        if ($choice == '1') {
            echo "\nDeleting existing chapters...\n";
            CourseChapter::where('course_id', $course->id)->delete();
            echo "✓ Deleted {$existingChapters} existing chapters.\n\n";
        } elseif ($choice == '3') {
            echo "Cancelled.\n";
            exit(0);
        }
    }
}

// Step 3: Define chapters based on HTML files
$chaptersPath = public_path('Flutter Online Book Version');
if (!File::exists($chaptersPath)) {
    echo "❌ ERROR: Chapters folder not found at: {$chaptersPath}\n";
    exit(1);
}

echo "Step 2: Reading chapter files from: {$chaptersPath}\n";

// Define all chapters with their titles
$chapters = [
    ['file' => 'introduction.html', 'number' => 0, 'title' => 'Introduction and Course Overview', 'session' => 0],
    ['file' => 'chapter-1.html', 'number' => 1, 'title' => 'Introduction to Mobile App Development', 'session' => 1],
    ['file' => 'chapter-2.html', 'number' => 2, 'title' => 'Dart Basics: Variables, Types, Operators, and I/O', 'session' => 2],
    ['file' => 'chapter-3.html', 'number' => 3, 'title' => 'Flutter Widgets and Architecture', 'session' => 3],
    ['file' => 'chapter-4.html', 'number' => 4, 'title' => 'Project Setup and Tooling', 'session' => 4],
    ['file' => 'chapter-5.html', 'number' => 5, 'title' => 'State Management Basics: setState and Patterns', 'session' => 5],
    ['file' => 'chapter-6.html', 'number' => 6, 'title' => 'Asynchronous Programming: Futures, Streams, and Isolates', 'session' => 6],
    ['file' => 'chapter-7.html', 'number' => 7, 'title' => 'Networking and REST APIs', 'session' => 7],
    ['file' => 'chapter-8.html', 'number' => 8, 'title' => 'Layout Deep Dive: Advanced Widgets, Constraints, and Responsive Design', 'session' => 8],
    ['file' => 'chapter-9.html', 'number' => 9, 'title' => 'Theming, Styling, and Custom Widgets', 'session' => 9],
    ['file' => 'chapter-10.html', 'number' => 10, 'title' => 'Navigation, Routing, and Deep Linking', 'session' => 10],
    ['file' => 'chapter-11.html', 'number' => 11, 'title' => 'Forms, Input Handling, and Validation', 'session' => 11],
    ['file' => 'chapter-12.html', 'number' => 12, 'title' => 'Advanced State Management: setState, InheritedWidget, Provider, and Scalable Patterns', 'session' => 12],
    ['file' => 'chapter-13.html', 'number' => 13, 'title' => 'Lists, Grids, and Dynamic Collections', 'session' => 13],
    ['file' => 'chapter-14.html', 'number' => 14, 'title' => 'Mini Project: Course Catalog App', 'session' => 14],
    ['file' => 'chapter-15.html', 'number' => 15, 'title' => 'Testing and CI/CD', 'session' => 15],
    ['file' => 'chapter-16.html', 'number' => 16, 'title' => 'Performance and Profiling', 'session' => 16],
    ['file' => 'chapter-17.html', 'number' => 17, 'title' => 'Security and Data Storage', 'session' => 17],
    ['file' => 'chapter-18.html', 'number' => 18, 'title' => 'Accessibility and Internationalization', 'session' => 18],
    ['file' => 'chapter-19.html', 'number' => 19, 'title' => 'Firebase Integration for Flutter', 'session' => 19],
    ['file' => 'chapter-20.html', 'number' => 20, 'title' => 'Publishing and Release Management', 'session' => 20],
    ['file' => 'chapter-21.html', 'number' => 21, 'title' => 'Project: Data Fetch System using Free Online API', 'session' => 21],
    ['file' => 'chapter-22.html', 'number' => 22, 'title' => 'Project: Simple Firebase CRUD', 'session' => 22],
    ['file' => 'chapter-23.html', 'number' => 23, 'title' => 'Project: Simple Quiz App', 'session' => 23],
    ['file' => 'chapter-24.html', 'number' => 24, 'title' => 'Project: Simple E-Commerce App', 'session' => 24],
];

// Function to extract chapter title from HTML
function extractChapterTitle($filePath) {
    if (!File::exists($filePath)) {
        return null;
    }
    
    $content = File::get($filePath);
    
    // Try to extract from <h1 class="chapter-title">
    if (preg_match('/<h1[^>]*class=["\']chapter-title["\'][^>]*>(.*?)<\/h1>/is', $content, $matches)) {
        return trim(strip_tags($matches[1]));
    }
    
    // Try to extract from <title> tag
    if (preg_match('/<title>(.*?)<\/title>/is', $content, $matches)) {
        $title = trim(strip_tags($matches[1]));
        // Remove "Chapter X - " prefix if present
        $title = preg_replace('/^Chapter \d+\s*-\s*/i', '', $title);
        // Remove " | Flutter & Dart E-Book" suffix if present
        $title = preg_replace('/\s*\|\s*Flutter.*$/i', '', $title);
        return trim($title);
    }
    
    return null;
}

// Function to extract description (learning objectives) from HTML
function extractDescription($filePath) {
    if (!File::exists($filePath)) {
        return null;
    }
    
    $content = File::get($filePath);
    
    // Try to extract learning objectives
    if (preg_match('/<section[^>]*class=["\']learning-objectives-section["\'][^>]*>.*?<p[^>]*class=["\']section-intro["\'][^>]*>(.*?)<\/p>/is', $content, $matches)) {
        $desc = trim(strip_tags($matches[1]));
        if (strlen($desc) > 200) {
            $desc = substr($desc, 0, 197) . '...';
        }
        return $desc;
    }
    
    return null;
}

echo "Step 3: Extracting chapter information from HTML files...\n\n";

$created = 0;
$skipped = 0;
$errors = 0;

DB::beginTransaction();

try {
    foreach ($chapters as $chapterData) {
        $filePath = $chaptersPath . '/' . $chapterData['file'];
        
        if (!File::exists($filePath)) {
            echo "⚠ WARNING: File not found: {$chapterData['file']}\n";
            $errors++;
            continue;
        }
        
        // Extract title if not provided
        $title = $chapterData['title'];
        if (!$title) {
            $title = extractChapterTitle($filePath);
            if (!$title) {
                echo "⚠ WARNING: Could not extract title from: {$chapterData['file']}\n";
                $title = "Chapter {$chapterData['number']}";
            }
        }
        
        // Extract description
        $description = extractDescription($filePath);
        
        // Check if chapter already exists
        $existing = CourseChapter::where('course_id', $course->id)
            ->where('chapter_number', $chapterData['number'])
            ->first();
        
        if ($existing) {
            echo "⊘ Skipped: Chapter {$chapterData['number']} - {$title} (already exists)\n";
            $skipped++;
            continue;
        }
        
        // Read HTML content (store path reference, not full content to save space)
        // We'll store the file path or a reference to it
        $content = "<!-- Chapter content available at: public/Flutter Online Book Version/{$chapterData['file']} -->";
        
        // Create chapter
        CourseChapter::create([
            'course_id' => $course->id,
            'chapter_number' => $chapterData['number'],
            'title' => $title,
            'description' => $description,
            'content' => $content,
            'session_number' => $chapterData['session'],
            'order' => $chapterData['number'],
            'is_published' => true,
        ]);
        
        echo "✓ Created: Chapter {$chapterData['number']} - {$title}\n";
        $created++;
    }
    
    DB::commit();
    
    echo "\n========================================\n";
    echo "Summary:\n";
    echo "  ✓ Created: {$created} chapters\n";
    echo "  ⊘ Skipped: {$skipped} chapters\n";
    if ($errors > 0) {
        echo "  ⚠ Errors: {$errors} files\n";
    }
    echo "========================================\n";
    echo "\n✓ Course curriculum populated successfully!\n";
    
} catch (\Exception $e) {
    DB::rollBack();
    echo "\n❌ ERROR: Failed to populate curriculum\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
    exit(1);
}

