<?php

namespace App\Services;

use App\Models\DeviceToken;
use App\Models\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class PushNotificationService
{
    /**
     * Send push notification to user's devices.
     *
     * @param int $userId
     * @param string $title
     * @param string $body
     * @param array $data
     * @return bool
     */
    public function sendToUser($userId, $title, $body, $data = [])
    {
        $deviceTokens = DeviceToken::where('user_id', $userId)
            ->where('is_active', true)
            ->get();

        if ($deviceTokens->isEmpty()) {
            return false;
        }

        $successCount = 0;
        foreach ($deviceTokens as $deviceToken) {
            if ($this->sendToDevice($deviceToken->device_token, $title, $body, $data)) {
                $successCount++;
                $deviceToken->update(['last_used_at' => now()]);
            }
        }

        return $successCount > 0;
    }

    /**
     * Send push notification to a specific device.
     *
     * @param string $deviceToken
     * @param string $title
     * @param string $body
     * @param array $data
     * @return bool
     */
    public function sendToDevice($deviceToken, $title, $body, $data = [])
    {
        $fcmServerKey = config('services.fcm.server_key');
        
        if (!$fcmServerKey) {
            Log::warning('FCM Server Key not configured');
            return false;
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'key=' . $fcmServerKey,
                'Content-Type' => 'application/json',
            ])->post('https://fcm.googleapis.com/fcm/send', [
                'to' => $deviceToken,
                'notification' => [
                    'title' => $title,
                    'body' => $body,
                    'sound' => 'default',
                    'badge' => 1,
                ],
                'data' => array_merge([
                    'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                ], $data),
                'priority' => 'high',
            ]);

            if ($response->successful()) {
                $result = $response->json();
                if (isset($result['success']) && $result['success'] == 1) {
                    return true;
                } else {
                    Log::warning('FCM send failed', ['response' => $result]);
                    return false;
                }
            } else {
                Log::error('FCM HTTP error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return false;
            }
        } catch (\Exception $e) {
            Log::error('FCM send exception: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Send push notification when a notification is created.
     *
     * @param Notification $notification
     * @return void
     */
    public function sendNotificationCreated($notification)
    {
        $this->sendToUser(
            $notification->user_id,
            $notification->title,
            $notification->message,
            [
                'notification_id' => $notification->id,
                'type' => $notification->type,
                'priority' => $notification->priority,
            ]
        );
    }
}

