<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class ApiCacheService
{
    /**
     * Cache duration in minutes.
     */
    const CACHE_DURATION = 60;
    const LONG_CACHE_DURATION = 1440; // 24 hours
    
    /**
     * Cache keys for different data types.
     */
    const CACHE_KEYS = [
        'books' => 'api:books:',
        'book_chapters' => 'api:book_chapters:',
        'courses' => 'api:courses:',
        'user_progress' => 'api:user_progress:',
        'assignments' => 'api:assignments:',
        'quiz_questions' => 'api:quiz_questions:',
    ];

    /**
     * Get cached data or store if not exists.
     *
     * @param string $key
     * @param callable $callback
     * @param int $duration
     * @return mixed
     */
    public static function remember(string $key, callable $callback, int $duration = self::CACHE_DURATION)
    {
        return Cache::remember($key, $duration, $callback);
    }

    /**
     * Get cached data with tags for better invalidation.
     *
     * @param string $key
     * @param array $tags
     * @param callable $callback
     * @param int $duration
     * @return mixed
     */
    public static function rememberWithTags(string $key, array $tags, callable $callback, int $duration = self::CACHE_DURATION)
    {
        if (config('cache.default') === 'redis') {
            return Cache::tags($tags)->remember($key, $duration, $callback);
        }
        
        return Cache::remember($key, $duration, $callback);
    }

    /**
     * Invalidate cache by key.
     *
     * @param string $key
     * @return bool
     */
    public static function forget(string $key): bool
    {
        return Cache::forget($key);
    }

    /**
     * Invalidate cache by tags.
     *
     * @param array $tags
     * @return void
     */
    public static function invalidateTags(array $tags): void
    {
        if (config('cache.default') === 'redis') {
            Cache::tags($tags)->flush();
        }
    }

    /**
     * Generate cache key for books listing.
     *
     * @param array $filters
     * @return string
     */
    public static function getBooksListKey(array $filters = []): string
    {
        $filterString = http_build_query($filters);
        return self::CACHE_KEYS['books'] . 'list:' . md5($filterString);
    }

    /**
     * Generate cache key for book details.
     *
     * @param int $bookId
     * @return string
     */
    public static function getBookDetailsKey(int $bookId): string
    {
        return self::CACHE_KEYS['books'] . "details:{$bookId}";
    }

    /**
     * Generate cache key for user progress.
     *
     * @param int $userId
     * @param string $type
     * @param int $itemId
     * @return string
     */
    public static function getUserProgressKey(int $userId, string $type, int $itemId): string
    {
        return self::CACHE_KEYS['user_progress'] . "{$userId}:{$type}:{$itemId}";
    }

    /**
     * Generate cache key for courses listing.
     *
     * @param int $instructorId
     * @param array $filters
     * @return string
     */
    public static function getCoursesListKey(int $instructorId = null, array $filters = []): string
    {
        $instructorPart = $instructorId ? "instructor:{$instructorId}:" : '';
        $filterString = http_build_query($filters);
        return self::CACHE_KEYS['courses'] . $instructorPart . 'list:' . md5($filterString);
    }

    /**
     * Generate cache key for assignments listing.
     *
     * @param int $instructorId
     * @param array $filters
     * @return string
     */
    public static function getAssignmentsListKey(int $instructorId, array $filters = []): string
    {
        $filterString = http_build_query($filters);
        return self::CACHE_KEYS['assignments'] . "instructor:{$instructorId}:list:" . md5($filterString);
    }

    /**
     * Clear all book-related caches for a specific book.
     *
     * @param int $bookId
     * @return void
     */
    public static function clearBookCaches(int $bookId): void
    {
        $keys = [
            self::getBookDetailsKey($bookId),
            self::CACHE_KEYS['book_chapters'] . $bookId,
        ];

        foreach ($keys as $key) {
            self::forget($key);
        }

        // Clear user progress caches for this book
        self::invalidateTags(['book_progress_' . $bookId]);
    }

    /**
     * Clear user-specific caches.
     *
     * @param int $userId
     * @return void
     */
    public static function clearUserCaches(int $userId): void
    {
        self::invalidateTags(['user_' . $userId]);
    }

    /**
     * Clear instructor-specific caches.
     *
     * @param int $instructorId
     * @return void
     */
    public static function clearInstructorCaches(int $instructorId): void
    {
        self::invalidateTags(['instructor_' . $instructorId]);
    }

    /**
     * Get cache statistics.
     *
     * @return array
     */
    public static function getCacheStats(): array
    {
        return [
            'driver' => config('cache.default'),
            'redis_enabled' => config('cache.default') === 'redis',
        ];
    }

    /**
     * Warm up common caches.
     *
     * @return void
     */
    public static function warmUpCommonCaches(): void
    {
        // Cache active books
        self::remember(
            self::CACHE_KEYS['books'] . 'active',
            function () {
                return DB::table('books')->where('is_active', true)->count();
            },
            self::LONG_CACHE_DURATION
        );

        // Cache active courses
        self::remember(
            self::CACHE_KEYS['courses'] . 'active',
            function () {
                return DB::table('courses')->where('is_active', true)->count();
            },
            self::LONG_CACHE_DURATION
        );
    }
}
