<?php

namespace App\Providers;

use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider as ServiceProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;

class RouteServiceProvider extends ServiceProvider
{
    /**
     * The path to the "home" route for your application.
     *
     * This is used by Laravel authentication to redirect users after login.
     *
     * @var string
     */
    public const HOME = '/home';

    /**
     * Define your route model bindings, pattern filters, etc.
     *
     * @return void
     */
    public function boot()
    {
        $this->configureRateLimiting();

        $this->routes(function () {
            Route::middleware('web')
                ->group(base_path('routes/web.php'));

            Route::prefix('api')
                ->middleware('api')
                ->group(base_path('routes/api.php'));
            
            // Note: routes/api/v1.php is already included in routes/api.php
            // No need to load it separately here
        });
    }

    /**
     * Configure the rate limiters for the application.
     *
     * @return void
     */
    protected function configureRateLimiting()
    {
        RateLimiter::for('api', function (Request $request) {
            return Limit::perMinute(60);
        });
        
        // Rate limiting for login attempts
        RateLimiter::for('login', function (Request $request) {
            // More lenient for development environments
            if (app()->environment(['local', 'testing'])) {
                return Limit::perMinute(60)->by($request->ip());
            }
            return Limit::perMinute(10)->by($request->ip());
        });
        
        // Rate limiting for password reset
        RateLimiter::for('password-reset', function (Request $request) {
            // More lenient for development environments
            if (app()->environment(['local', 'testing'])) {
                return Limit::perMinute(30)->by($request->ip());
            }
            return Limit::perMinute(5)->by($request->ip());
        });
        
        // Rate limiting for API authentication endpoints
        RateLimiter::for('api-auth', function (Request $request) {
            // Use user ID if authenticated, otherwise use IP
            $key = $request->user() ? $request->user()->id : $request->ip();
            
            // More lenient for development environments
            if (app()->environment(['local', 'testing'])) {
                return Limit::perMinute(60)->by($key);
            }
            return Limit::perMinute(30)->by($key);
        });
        
        // Rate limiting for API endpoints - more generous for authenticated users
        RateLimiter::for('api-endpoints', function (Request $request) {
            // Authenticated users get higher limits
            if ($request->user()) {
                return Limit::perMinute(120)->by($request->user()->id);
            }
            // Unauthenticated requests (shouldn't happen with auth middleware, but just in case)
            return Limit::perMinute(60)->by($request->ip());
        });
    }
}
