<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Assignment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'course_id',
        'chapter_id',
        'title',
        'description',
        'instructions',
        'due_date',
        'due_time',
        'total_points',
        'allow_late_submission',
        'late_penalty_percentage',
        'is_published',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'due_date' => 'date',
        'total_points' => 'decimal:2',
        'allow_late_submission' => 'boolean',
        'is_published' => 'boolean',
    ];

    /**
     * Get the course that owns this assignment.
     */
    public function course()
    {
        return $this->belongsTo(Course::class, 'course_id');
    }

    /**
     * Get the chapter this assignment is associated with.
     */
    public function chapter()
    {
        return $this->belongsTo(CourseChapter::class, 'chapter_id');
    }

    /**
     * Get the submissions for this assignment.
     */
    public function submissions()
    {
        return $this->hasMany(AssignmentSubmission::class, 'assignment_id');
    }

    /**
     * Check if the assignment is past due.
     */
    public function isPastDue()
    {
        $dueDateTime = $this->due_date;
        if ($this->due_time) {
            // Handle due_time as string (H:i format from database time column)
            $dueTimeStr = is_string($this->due_time) ? $this->due_time : (is_object($this->due_time) ? $this->due_time->format('H:i') : $this->due_time);
            $dueDateTime = Carbon::parse($this->due_date->format('Y-m-d') . ' ' . $dueTimeStr);
        } else {
            $dueDateTime = $this->due_date->endOfDay();
        }
        return Carbon::now()->greaterThan($dueDateTime);
    }

    /**
     * Get the due date and time as a Carbon instance.
     */
    public function getDueDateTimeAttribute()
    {
        if ($this->due_time) {
            // Handle due_time as string (H:i format from database time column)
            $dueTimeStr = is_string($this->due_time) ? $this->due_time : (is_object($this->due_time) ? $this->due_time->format('H:i') : $this->due_time);
            return Carbon::parse($this->due_date->format('Y-m-d') . ' ' . $dueTimeStr);
        }
        return $this->due_date->endOfDay();
    }
}

