<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SanitizeInput
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // Skip sanitization for file uploads and specific content types
        if ($request->hasFile('file') || $request->hasFile('submission_file') || 
            $request->hasFile('video_file') || $request->isJson()) {
            return $next($request);
        }

        // Sanitize input data
        $input = $request->all();
        
        foreach ($input as $key => $value) {
            if (is_string($value)) {
                // Remove HTML tags and encode special characters
                $input[$key] = $this->sanitizeString($value);
            } elseif (is_array($value)) {
                $input[$key] = $this->sanitizeArray($value);
            }
        }

        // Replace the request input with sanitized data
        $request->merge($input);

        return $next($request);
    }

    /**
     * Sanitize a string input.
     *
     * @param string $value
     * @return string
     */
    private function sanitizeString($value)
    {
        // Trim whitespace
        $value = trim($value);
        
        // Remove potentially dangerous HTML tags but allow basic formatting
        $allowedTags = '<p><br><strong><em><u><ol><ul><li>';
        $value = strip_tags($value, $allowedTags);
        
        // Encode special characters to prevent XSS
        $value = htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
        
        // Remove control characters except newlines and tabs
        $value = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $value);
        
        return $value;
    }

    /**
     * Sanitize an array input.
     *
     * @param array $array
     * @return array
     */
    private function sanitizeArray($array)
    {
        foreach ($array as $key => $value) {
            if (is_string($value)) {
                $array[$key] = $this->sanitizeString($value);
            } elseif (is_array($value)) {
                $array[$key] = $this->sanitizeArray($value);
            }
        }
        
        return $array;
    }
}
