<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class CheckInstallation
{
    /**
     * Check if request is from localhost.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return bool
     */
    private function isLocalhost(Request $request)
    {
        $host = $request->getHost();
        $ip = $request->ip();
        
        // Check if host is localhost or local IPs
        return in_array($host, ['localhost', '127.0.0.1', '::1']) 
            || in_array($ip, ['127.0.0.1', '::1', 'localhost'])
            || strpos($host, '127.0.0.1') === 0
            || strpos($ip, '127.0.0.1') === 0;
    }
    
    /**
     * Check if database connection exists and tables are created.
     *
     * @return bool
     */
    private function isDatabaseConfigured()
    {
        try {
            // Check if database connection exists
            DB::connection()->getPdo();
            
            // Check if users table exists (indicates database is set up)
            if (Schema::hasTable('users')) {
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            // Database connection doesn't exist or not configured
            return false;
        }
    }
    
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // Skip all checks for localhost - allow normal access for local development
        if ($this->isLocalhost($request)) {
            return $next($request);
        }
        
        $isSetupRoute = $request->routeIs('setup.*');
        
        // Allow setup routes to proceed immediately if lock file doesn't exist
        // This prevents database connection errors from blocking setup
        $isInstalled = File::exists(storage_path('app/installed.lock'));
        
        if ($isSetupRoute) {
            // If installed lock exists, block setup
            if ($isInstalled) {
                abort(404, 'Application is already installed.');
            }
            // Allow setup to proceed
            return $next($request);
        }
        
        // For non-setup routes, check if installation is needed
        if (!$isInstalled) {
            // Try to check database, but don't fail if it's not configured yet
            try {
                $isDatabaseConfigured = $this->isDatabaseConfigured();
                
                // If database is already configured, consider it installed
                if ($isDatabaseConfigured) {
                    // Create installed lock file if database exists but lock doesn't
                    File::put(storage_path('app/installed.lock'), date('Y-m-d H:i:s'));
                    $isInstalled = true;
                }
            } catch (\Exception $e) {
                // Database not configured yet - that's okay, allow setup
                $isInstalled = false;
            }
        }
        
        // If not installed/configured and not accessing setup routes, redirect to setup
        if (!$isInstalled && !$isSetupRoute) {
            // Don't redirect if accessing login/auth routes (they'll be blocked later anyway)
            if (!$request->routeIs('login.*') && !$request->routeIs('password.*')) {
                return redirect()->route('setup.welcome');
            }
        }
        
        return $next($request);
    }
}

