<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Course;
use App\Models\Enrollment;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\ClassSessionVideo;
use App\Models\Certificate;
use App\Models\FeePayment;
use App\Models\CourseChapter;
use App\Models\ChapterProgress;
use Illuminate\Support\Facades\Storage;

class StudentController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:student']);
    }

    /**
     * Display the student dashboard.
     *
     * @return \Illuminate\View\View
     */
    public function dashboard()
    {
        $student = Auth::user();
        
        // Get enrolled courses
        $enrollments = Enrollment::where('student_id', $student->id)
            ->with('course')
            ->get();
        
        $totalCourses = $enrollments->count();
        
        // Get assignments statistics
        $courseIds = $enrollments->pluck('course_id');
        $totalAssignments = Assignment::whereIn('course_id', $courseIds)
            ->where('is_published', true)
            ->count();
        
        $pendingAssignments = Assignment::whereIn('course_id', $courseIds)
            ->where('is_published', true)
            ->whereDoesntHave('submissions', function($q) use ($student) {
                $q->where('student_id', $student->id)
                  ->whereIn('status', ['submitted', 'graded']);
            })
            ->where('due_date', '>=', now()->toDateString())
            ->count();
        
        $gradedAssignments = AssignmentSubmission::where('student_id', $student->id)
            ->where('status', 'graded')
            ->count();
        
        // Get recent activities
        $recentCourses = $enrollments->take(5);
        
        $recentAssignments = Assignment::whereIn('course_id', $courseIds)
            ->where('is_published', true)
            ->with(['course', 'submissions' => function($q) use ($student) {
                $q->where('student_id', $student->id);
            }])
            ->orderBy('due_date', 'asc')
            ->limit(5)
            ->get();
        
        // Calculate fee information for alerts
        $feeAlerts = [];
        foreach ($enrollments as $enrollment) {
            if ($enrollment->course) {
                $courseFee = $enrollment->course->fees;
                $paidAmount = FeePayment::where('student_id', $student->id)
                    ->where(function($q) use ($enrollment) {
                        $q->where('course_id', $enrollment->course_id);
                        if ($enrollment->course_offering_id) {
                            $q->orWhere('course_offering_id', $enrollment->course_offering_id);
                        }
                    })
                    ->sum('amount');
                
                $balance = $courseFee - $paidAmount;
                
                if ($balance > 0) {
                    $feeAlerts[] = [
                        'course' => $enrollment->course,
                        'balance' => $balance,
                        'paid' => $paidAmount,
                        'total' => $courseFee,
                    ];
                }
            }
        }
        
        $stats = [
            'total_courses' => $totalCourses,
            'total_assignments' => $totalAssignments,
            'pending_assignments' => $pendingAssignments,
            'graded_assignments' => $gradedAssignments,
        ];
        
        return view('student.dashboard', compact('stats', 'recentCourses', 'recentAssignments', 'feeAlerts'));
    }
}

