<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Course;
use App\Models\Enrollment;
use App\Models\CourseChapter;
use App\Models\ChapterProgress;

class CourseController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:student']);
    }

    /**
     * Display a listing of enrolled courses.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $student = Auth::user();
        
        $enrollments = Enrollment::where('student_id', $student->id)
            ->with(['course.instructor', 'courseOffering'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        return view('student.courses.index', compact('enrollments'));
    }

    /**
     * Display the specified course.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $student = Auth::user();
        
        // Verify student is enrolled
        $enrollment = Enrollment::where('student_id', $student->id)
            ->where('course_id', $id)
            ->firstOrFail();
        
        $course = Course::with(['instructor', 'chapters' => function($q) {
            $q->where('is_published', true)->orderBy('order');
        }])
            ->findOrFail($id);
        
        // Get student's progress for each chapter
        $chapterProgress = ChapterProgress::where('student_id', $student->id)
            ->where('course_id', $course->id)
            ->get()
            ->keyBy('chapter_id');
        
        // Calculate overall progress
        $totalChapters = $course->chapters->count();
        $completedChapters = $chapterProgress->where('status', 'completed')->count();
        $progressPercentage = $totalChapters > 0 ? ($completedChapters / $totalChapters) * 100 : 0;
        
        return view('student.courses.show', compact('course', 'enrollment', 'chapterProgress', 'progressPercentage'));
    }
}

