<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\Enrollment;
use Carbon\Carbon;

class AssignmentController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:student']);
    }

    /**
     * Display a listing of assignments for enrolled courses.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $student = Auth::user();
        
        // Get enrolled course IDs
        $courseIds = Enrollment::where('student_id', $student->id)->pluck('course_id');
        
        $query = Assignment::whereIn('course_id', $courseIds)
            ->where('is_published', true)
            ->with(['course', 'chapter', 'submissions' => function($q) use ($student) {
                $q->where('student_id', $student->id);
            }]);
        
        // Filter by course
        if ($request->has('course_id') && $request->course_id) {
            $query->where('course_id', $request->course_id);
        }
        
        // Filter by status
        if ($request->has('status') && $request->status) {
            if ($request->status == 'pending') {
                $query->whereDoesntHave('submissions', function($q) use ($student) {
                    $q->where('student_id', $student->id)
                      ->whereIn('status', ['submitted', 'graded']);
                })->where('due_date', '>=', now()->toDateString());
            } elseif ($request->status == 'submitted') {
                $query->whereHas('submissions', function($q) use ($student) {
                    $q->where('student_id', $student->id)
                      ->where('status', 'submitted');
                });
            } elseif ($request->status == 'graded') {
                $query->whereHas('submissions', function($q) use ($student) {
                    $q->where('student_id', $student->id)
                      ->where('status', 'graded');
                });
            } elseif ($request->status == 'overdue') {
                $query->where('due_date', '<', now()->toDateString())
                    ->whereDoesntHave('submissions', function($q) use ($student) {
                        $q->where('student_id', $student->id)
                          ->whereIn('status', ['submitted', 'graded']);
                    });
            }
        }
        
        $assignments = $query->orderBy('due_date', 'asc')->paginate(20);
        
        // Get courses for filter
        $courses = \App\Models\Course::whereIn('id', $courseIds)
            ->where('is_active', true)
            ->orderBy('name', 'asc')
            ->get();
        
        return view('student.assignments.index', compact('assignments', 'courses'));
    }

    /**
     * Display the specified assignment.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $student = Auth::user();
        
        // Get enrolled course IDs
        $courseIds = Enrollment::where('student_id', $student->id)->pluck('course_id');
        
        $assignment = Assignment::where('id', $id)
            ->whereIn('course_id', $courseIds)
            ->where('is_published', true)
            ->with(['course', 'chapter'])
            ->firstOrFail();
        
        // Get student's submission if exists
        $submission = AssignmentSubmission::where('assignment_id', $id)
            ->where('student_id', $student->id)
            ->first();
        
        return view('student.assignments.show', compact('assignment', 'submission'));
    }

    /**
     * Store a new assignment submission.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function submit(Request $request, $id)
    {
        try {
            $student = Auth::user();
            
            // Verify student is enrolled in the course
            $courseIds = Enrollment::where('student_id', $student->id)->pluck('course_id');
            
            $assignment = Assignment::where('id', $id)
                ->whereIn('course_id', $courseIds)
                ->where('is_published', true)
                ->firstOrFail();
            
            // Validate that at least one of submission_text or submission_file is provided
            $validated = $request->validate([
                'submission_text' => 'nullable|string',
                'submission_file' => 'nullable|file|max:10240', // 10MB max
            ], [
                'submission_text.nullable' => 'You must provide either submission text or a file.',
                'submission_file.nullable' => 'You must provide either submission text or a file.',
            ]);
            
            // Ensure at least one submission field is provided (for new submissions)
            // For updates, allow if existing submission already has content
            if (!$existingSubmission) {
                // For new submissions, require at least one field
                if (empty($validated['submission_text']) && !$request->hasFile('submission_file')) {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors(['submission' => 'You must provide either submission text or a file.']);
                }
            } else {
                // For updates, ensure at least one field is provided or existing submission has content
                $hasNewContent = !empty($validated['submission_text']) || $request->hasFile('submission_file');
                $hasExistingContent = !empty($existingSubmission->submission_text) || !empty($existingSubmission->submission_file);
                
                if (!$hasNewContent && !$hasExistingContent) {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors(['submission' => 'You must provide either submission text or a file.']);
                }
            }
            
            // Check if assignment is past due and if late submissions are allowed
            $dueDateTime = $assignment->due_date;
            if ($assignment->due_time) {
                // Handle due_time as string (H:i format) or as time
                $dueTimeStr = is_string($assignment->due_time) ? $assignment->due_time : $assignment->due_time->format('H:i');
                $dueDateTime = Carbon::parse($assignment->due_date->format('Y-m-d') . ' ' . $dueTimeStr);
            } else {
                $dueDateTime = $assignment->due_date->endOfDay();
            }
            
            $isLate = Carbon::now()->greaterThan($dueDateTime);
            
            // Check if late submission is allowed
            if ($isLate && !$assignment->allow_late_submission) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['due_date' => 'This assignment does not allow late submissions.']);
            }
            
            // Check if submission already exists
            $existingSubmission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();
            
            // Prevent resubmission if already graded
            if ($existingSubmission && $existingSubmission->status == 'graded') {
                return redirect()->route('student.assignments.show', $assignment->id)
                    ->withErrors(['error' => 'This assignment has already been graded. Resubmission is not allowed.']);
            }
            
            // Handle file upload
            $filePath = null;
            if ($request->hasFile('submission_file')) {
                // Delete old file if exists and a new file is being uploaded
                if ($existingSubmission && $existingSubmission->submission_file) {
                    Storage::disk('public')->delete($existingSubmission->submission_file);
                }
                $filePath = $request->file('submission_file')->store('assignment-submissions', 'public');
            } else {
                // Keep existing file if no new file is uploaded
                $filePath = $existingSubmission ? $existingSubmission->submission_file : null;
            }
            
            // Prepare submission data
            // If updating, preserve existing text if new text is not provided
            $submissionText = null;
            if (!empty($validated['submission_text'])) {
                $submissionText = $validated['submission_text'];
            } elseif ($existingSubmission) {
                $submissionText = $existingSubmission->submission_text;
            }
            
            $submissionData = [
                'course_id' => $assignment->course_id,
                'submission_text' => $submissionText,
                'submission_file' => $filePath,
                'submitted_at' => now(),
                'is_late' => $isLate,
                'status' => 'submitted',
            ];
            
            // Update or create submission
            if ($existingSubmission) {
                $existingSubmission->update($submissionData);
                $submission = $existingSubmission;
            } else {
                $submission = AssignmentSubmission::create(array_merge([
                    'assignment_id' => $assignment->id,
                    'student_id' => $student->id,
                ], $submissionData));
            }
            
            return redirect()->route('student.assignments.show', $assignment->id)
                ->with('success', 'Assignment submitted successfully!');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return redirect()->route('student.assignments.index')
                ->withErrors(['error' => 'Assignment not found or you do not have access to it.']);
        } catch (\Exception $e) {
            \Log::error('Assignment submission error: ' . $e->getMessage(), [
                'assignment_id' => $id,
                'student_id' => Auth::id(),
                'exception' => $e
            ]);
            
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while submitting the assignment. Please try again.']);
        }
    }
}

