<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Course;
use App\Models\Enrollment;
use App\Models\ClassSessionVideo;
use App\Models\CourseOffering;
use App\Models\User;

class InstructorController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:instructor']);
    }

    /**
     * Display the instructor dashboard.
     *
     * @return \Illuminate\View\View
     */
    public function dashboard()
    {
        $instructor = Auth::user();
        
        // Get statistics for instructor's courses
        $totalCourses = Course::where('instructor_id', $instructor->id)
            ->where('is_active', true)
            ->count();
        
        $totalStudents = Enrollment::whereIn('course_id', function($query) use ($instructor) {
            $query->select('id')
                ->from('courses')
                ->where('instructor_id', $instructor->id);
        })->distinct('student_id')->count();
        
        $totalEnrollments = Enrollment::whereIn('course_id', function($query) use ($instructor) {
            $query->select('id')
                ->from('courses')
                ->where('instructor_id', $instructor->id);
        })->count();
        
        $totalVideos = ClassSessionVideo::whereIn('course_id', function($query) use ($instructor) {
            $query->select('id')
                ->from('courses')
                ->where('instructor_id', $instructor->id);
        })->where('is_published', true)->count();
        
        // Get recent activities
        $recentCourses = Course::where('instructor_id', $instructor->id)
            ->where('is_active', true)
            ->orderBy('updated_at', 'desc')
            ->limit(5)
            ->get();
        
        $recentEnrollments = Enrollment::whereIn('course_id', function($query) use ($instructor) {
            $query->select('id')
                ->from('courses')
                ->where('instructor_id', $instructor->id);
        })->with(['student', 'course'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        $recentVideos = ClassSessionVideo::whereIn('course_id', function($query) use ($instructor) {
            $query->select('id')
                ->from('courses')
                ->where('instructor_id', $instructor->id);
        })->with('course')
            ->where('is_published', true)
            ->orderBy('session_date', 'desc')
            ->limit(5)
            ->get();
        
        $stats = [
            'total_courses' => $totalCourses,
            'total_students' => $totalStudents,
            'total_enrollments' => $totalEnrollments,
            'total_videos' => $totalVideos,
        ];
        
        return view('instructor.dashboard', compact('stats', 'recentCourses', 'recentEnrollments', 'recentVideos'));
    }
}

