<?php

namespace App\Http\Controllers\Instructor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Course;
use App\Models\Enrollment;

class StudentController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:instructor']);
    }

    /**
     * Display a listing of students enrolled in instructor's courses.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $instructor = Auth::user();
        
        // Get all course IDs for this instructor
        $courseIds = Course::where('instructor_id', $instructor->id)
            ->where('is_active', true)
            ->pluck('id');
        
        // Get unique students enrolled in these courses
        $studentIds = Enrollment::whereIn('course_id', $courseIds)
            ->distinct('student_id')
            ->pluck('student_id');
        
        $query = User::whereIn('id', $studentIds)
            ->where('role', 'student')
            ->withCount(['enrollments' => function($q) use ($courseIds) {
                $q->whereIn('course_id', $courseIds);
            }]);
        
        // Search filter
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('school_id', 'like', "%{$search}%");
            });
        }
        
        $students = $query->orderBy('name', 'asc')->paginate(20);
        
        // Get courses for filter dropdown
        $courses = Course::where('instructor_id', $instructor->id)
            ->where('is_active', true)
            ->orderBy('name', 'asc')
            ->get();
        
        return view('instructor.students.index', compact('students', 'courses'));
    }

    /**
     * Display the specified student.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $instructor = Auth::user();
        
        // Get instructor's course IDs
        $courseIds = Course::where('instructor_id', $instructor->id)
            ->where('is_active', true)
            ->pluck('id');
        
        $student = User::where('id', $id)
            ->where('role', 'student')
            ->firstOrFail();
        
        // Get enrollments for this student in instructor's courses
        $enrollments = Enrollment::where('student_id', $student->id)
            ->whereIn('course_id', $courseIds)
            ->with(['course', 'courseOffering'])
            ->get();
        
        return view('instructor.students.show', compact('student', 'enrollments'));
    }
}

