<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use Carbon\Carbon;

class ResetPasswordController extends Controller
{
    /**
     * Show the password reset form.
     *
     * @param  string  $token
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\View\View
     */
    public function showResetForm(Request $request, $token = null)
    {
        $email = $request->query('email');
        $role = $request->query('role', 'student');
        
        return view('auth.passwords.reset', [
            'token' => $token,
            'email' => $email,
            'role' => $role,
        ]);
    }

    /**
     * Reset the user's password.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reset(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|string|min:8|confirmed',
        ], [
            'password.required' => 'Please enter a new password.',
            'password.min' => 'Password must be at least 8 characters.',
            'password.confirmed' => 'Password confirmation does not match.',
        ]);

        $email = $request->email;
        $role = $request->input('role', 'student');
        $token = $request->token;

        // Find password reset record
        $passwordReset = DB::table('password_resets')
            ->where('email', $email)
            ->first();

        if (!$passwordReset) {
            return back()->withErrors(['email' => 'Invalid or expired reset token.'])
                ->withInput($request->only('email'));
        }

        // Check if token matches (within 60 minutes)
        $tokenAge = Carbon::parse($passwordReset->created_at)->diffInMinutes(Carbon::now());
        
        if ($tokenAge > 60) {
            DB::table('password_resets')->where('email', $email)->delete();
            return back()->withErrors(['email' => 'This password reset link has expired. Please request a new one.'])
                ->withInput($request->only('email'));
        }

        // Verify token - the token in DB is hashed with bcrypt
        if (!Hash::check($token, $passwordReset->token)) {
            return back()->withErrors(['email' => 'Invalid reset token.'])
                ->withInput($request->only('email'));
        }

        // Find user
        $user = User::where('email', $email)
            ->where('role', $role)
            ->first();

        if (!$user) {
            return back()->withErrors(['email' => 'User not found.'])
                ->withInput($request->only('email'));
        }

        // Update password
        $user->password = Hash::make($request->password);
        $user->save();

        // Delete used token
        DB::table('password_resets')->where('email', $email)->delete();

        // Redirect to login with success message
        $loginRoute = $role === 'admin' ? 'admin.login' : ($role === 'instructor' ? 'instructor.login' : 'login');
        
        return redirect()->route($loginRoute)
            ->with('status', 'Your password has been reset successfully! You can now login with your new password.');
    }
}

