<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use App\Models\User;

class LoginController extends Controller
{
    /**
     * Show the student login form (base route)
     */
    public function showStudentLoginForm()
    {
        // If user is already authenticated, redirect to their dashboard
        if (Auth::check()) {
            return $this->redirectAfterLogin(Auth::user());
        }
        
        return view('auth.login', ['role' => 'student']);
    }

    /**
     * Show the admin login form
     */
    public function showAdminLoginForm()
    {
        // If user is already authenticated, redirect to their dashboard
        if (Auth::check()) {
            return $this->redirectAfterLogin(Auth::user());
        }
        
        return view('auth.login', ['role' => 'admin']);
    }

    /**
     * Show the instructor login form
     */
    public function showInstructorLoginForm()
    {
        // If user is already authenticated, redirect to their dashboard
        if (Auth::check()) {
            return $this->redirectAfterLogin(Auth::user());
        }
        
        return view('auth.login', ['role' => 'instructor']);
    }

    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        // Determine role from route
        $role = $this->determineRoleFromRoute($request);
        
        $request->validate([
            'login' => 'required|string',
            'password' => 'required|string',
        ]);

        $request->merge(['role' => $role]);
        $user = $this->authenticateUser($request);
        
        if ($user) {
            // Use remember me functionality - extends session to 2 weeks if checked
            $remember = $request->filled('remember');
            Auth::login($user, $remember);
            
            // Regenerate session ID for security
            $request->session()->regenerate();
            
            // If remember me is checked, set a longer session lifetime
            if ($remember) {
                // Set session to expire in 2 weeks (20160 minutes)
                $request->session()->put('remember_me', true);
            }
            
            // Redirect based on role
            return $this->redirectAfterLogin($user);
        }

        throw ValidationException::withMessages([
            'login' => ['The provided credentials are incorrect.'],
        ]);
    }

    /**
     * Determine role from the route
     */
    protected function determineRoleFromRoute(Request $request)
    {
        $path = $request->path();
        
        if (strpos($path, 'admin') !== false) {
            return 'admin';
        } elseif (strpos($path, 'instructor') !== false) {
            return 'instructor';
        } else {
            return 'student';
        }
    }

    /**
     * Authenticate user based on role and login type
     */
    protected function authenticateUser(Request $request)
    {
        $login = $request->login;
        $password = $request->password;
        $role = $request->role;
        
        // If role is student, authenticate by school_id
        if ($role === 'student') {
            $user = User::where('school_id', $login)
                ->where('role', 'student')
                ->first();
        } else {
            // For admin and instructor, authenticate by email
            $user = User::where('email', $login)
                ->where('role', $role)
                ->first();
        }
        
        // Check if user exists and password is correct
        if ($user && Hash::check($password, $user->password)) {
            return $user;
        }
        
        return null;
    }

    /**
     * Redirect user after successful login
     */
    protected function redirectAfterLogin($user)
    {
        switch ($user->role) {
            case 'admin':
                return redirect()->route('admin.dashboard');
            case 'instructor':
                return redirect()->route('instructor.dashboard');
            case 'student':
                return redirect()->route('student.dashboard');
            default:
                return redirect()->route('home');
        }
    }

    /**
     * Handle logout request
     */
    public function logout(Request $request)
    {
        $user = Auth::user();
        $role = $user ? $user->role : 'student';
        
        Auth::logout();
        
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        // Redirect to appropriate login page based on role
        if ($role === 'admin') {
            return redirect()->route('admin.login');
        } elseif ($role === 'instructor') {
            return redirect()->route('instructor.login');
        } else {
            return redirect()->route('student.login');
        }
    }
}
