<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Mail;
use App\Models\User;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ForgotPasswordController extends Controller
{
    /**
     * Show the form for requesting a password reset link.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\View\View
     */
    public function showLinkRequestForm(Request $request)
    {
        // Get role from query parameter or default to student
        $role = $request->query('role', 'student');
        
        // Validate role
        if (!in_array($role, ['student', 'instructor', 'admin'])) {
            $role = 'student';
        }
        
        return view('auth.passwords.email', ['role' => $role]);
    }

    /**
     * Send a reset link to the given user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function sendResetLinkEmail(Request $request)
    {
        $role = $request->input('role', 'student');
        
        $request->validate([
            'email' => 'required|email',
        ], [
            'email.required' => 'Please enter your email address.',
            'email.email' => 'Please enter a valid email address.',
        ]);

        // Find user by email and role
        $user = User::where('email', $request->email)
            ->where('role', $role)
            ->first();

        if (!$user) {
            return back()->withErrors(['email' => 'We could not find a user with that email address for this role.'])
                ->withInput($request->only('email'));
        }

        // Generate reset token
        $token = Str::random(64);
        
        // Delete old tokens for this email
        DB::table('password_resets')->where('email', $request->email)->delete();
        
        // Insert new token
        DB::table('password_resets')->insert([
            'email' => $request->email,
            'token' => bcrypt($token),
            'created_at' => Carbon::now(),
        ]);

        // Send reset email
        $resetUrl = url("/password/reset/{$token}?email=" . urlencode($request->email) . "&role=" . $role);
        
        try {
            // Ensure mail config is loaded from settings
            $this->loadMailConfig();
            
            Mail::send('emails.password-reset', [
                'user' => $user,
                'resetUrl' => $resetUrl,
                'role' => $role,
            ], function ($message) use ($user, $role) {
                $fromAddress = \App\Models\Setting::get('mail_from_address', 'noreply@afrilen.com');
                $fromName = \App\Models\Setting::get('mail_from_name', 'Afrilen Technologies');
                
                $message->to($user->email)
                    ->from($fromAddress, $fromName)
                    ->subject('Reset Password - ' . \App\Models\Setting::get('system_name', 'Afrilen Technologies'));
            });

            return back()->with('status', 'We have emailed your password reset link!');
        } catch (\Exception $e) {
            \Log::error('Password reset email error: ' . $e->getMessage());
            return back()->withErrors(['email' => 'Failed to send email. Please check your email settings or try again later.'])
                ->withInput($request->only('email'));
        }
    }
    
    /**
     * Load mail configuration from settings.
     *
     * @return void
     */
    private function loadMailConfig()
    {
        try {
            if (\Schema::hasTable('settings')) {
                $mailer = \App\Models\Setting::get('mail_mailer', 'smtp');
                $host = \App\Models\Setting::get('mail_host');
                $port = \App\Models\Setting::get('mail_port', '587');
                $username = \App\Models\Setting::get('mail_username');
                $password = \App\Models\Setting::getEncrypted('mail_password');
                $encryption = \App\Models\Setting::get('mail_encryption', 'tls');

                if ($host && $port) {
                    config([
                        'mail.default' => $mailer,
                        'mail.mailers.smtp.host' => $host,
                        'mail.mailers.smtp.port' => $port,
                        'mail.mailers.smtp.username' => $username,
                        'mail.mailers.smtp.password' => $password,
                        'mail.mailers.smtp.encryption' => ($encryption === 'null' || empty($encryption)) ? null : $encryption,
                        'mail.from.address' => \App\Models\Setting::get('mail_from_address', 'noreply@afrilen.com'),
                        'mail.from.name' => \App\Models\Setting::get('mail_from_name', 'Afrilen Technologies'),
                    ]);
                }
            }
        } catch (\Exception $e) {
            // Use default config if settings can't be loaded
            \Log::warning('Failed to load mail config for password reset: ' . $e->getMessage());
        }
    }
}

