<?php

namespace App\Http\Controllers\Api\V1\Student;

use App\Http\Controllers\Api\V1\BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Certificate;

class CertificateController extends BaseController
{
    /**
     * Get list of certificates for the student.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            
            $certificates = Certificate::where('student_id', $user->id)
                ->where('is_active', true)
                ->with(['course', 'courseOffering'])
                ->orderBy('issued_date', 'desc')
                ->get();
            
            $data = $certificates->map(function($certificate) {
                return [
                    'id' => $certificate->id,
                    'title' => $certificate->title,
                    'certificate_type' => $certificate->certificate_type,
                    'file_path' => $certificate->file_path ? asset('storage/' . $certificate->file_path) : null,
                    'file_name' => $certificate->file_name,
                    'file_size' => $certificate->file_size,
                    'issued_date' => $certificate->issued_date,
                    'description' => $certificate->description,
                    'course' => $certificate->course ? [
                        'id' => $certificate->course->id,
                        'name' => $certificate->course->name,
                    ] : null,
                    'course_offering' => $certificate->courseOffering ? [
                        'id' => $certificate->courseOffering->id,
                        'name' => $certificate->courseOffering->name,
                    ] : null,
                    'download_count' => $certificate->download_count ?? 0,
                    'created_at' => $certificate->created_at,
                    'updated_at' => $certificate->updated_at,
                ];
            });
            
            return $this->success($data);
        } catch (\Exception $e) {
            Log::error('Student Certificates API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching certificates.', null, 500);
        }
    }
    
    /**
     * Get certificate details.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $user = Auth::user();
            
            $certificate = Certificate::where('id', $id)
                ->where('student_id', $user->id)
                ->where('is_active', true)
                ->with(['course', 'courseOffering'])
                ->firstOrFail();
            
            // Increment download count
            $certificate->increment('download_count');
            
            return $this->success([
                'id' => $certificate->id,
                'title' => $certificate->title,
                'certificate_type' => $certificate->certificate_type,
                'file_path' => $certificate->file_path ? asset('storage/' . $certificate->file_path) : null,
                'file_name' => $certificate->file_name,
                'file_size' => $certificate->file_size,
                'issued_date' => $certificate->issued_date,
                'description' => $certificate->description,
                'course' => $certificate->course ? [
                    'id' => $certificate->course->id,
                    'name' => $certificate->course->name,
                ] : null,
                'course_offering' => $certificate->courseOffering ? [
                    'id' => $certificate->courseOffering->id,
                    'name' => $certificate->courseOffering->name,
                ] : null,
                'download_count' => $certificate->download_count ?? 0,
                'created_at' => $certificate->created_at,
                'updated_at' => $certificate->updated_at,
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Certificate not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Student Certificate API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching certificate details.', null, 500);
        }
    }
}

