<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Http\Controllers\Api\V1\BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use App\Models\ClassSessionVideo;
use App\Models\Course;

class VideoController extends BaseController
{
    /**
     * Get list of videos for instructor's courses.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $instructor = Auth::user();
            
            // Get instructor's course IDs
            $courseIds = Course::where('instructor_id', $instructor->id)
                ->where('is_active', true)
                ->pluck('id');
            
            $query = ClassSessionVideo::whereIn('course_id', $courseIds)
                ->with(['course', 'courseOffering']);
            
            // Filter by course
            if ($request->has('course_id') && $request->course_id) {
                $query->where('course_id', $request->course_id);
            }
            
            // Filter by published status
            if ($request->has('published') && $request->published !== '') {
                $query->where('is_published', $request->published == '1');
            }
            
            $videos = $query->orderBy('session_date', 'desc')
                ->paginate($request->get('per_page', 20));
            
            $data = $videos->map(function($video) {
                return [
                    'id' => $video->id,
                    'title' => $video->title,
                    'description' => $video->description,
                    'video_url' => $video->mega_storage_link ?? $video->video_url,
                    'download_key' => $video->download_key,
                    'thumbnail_url' => $video->thumbnail_url,
                    'session_date' => $video->session_date,
                    'is_published' => $video->is_published,
                    'course' => [
                        'id' => $video->course->id ?? null,
                        'name' => $video->course->name ?? 'N/A',
                    ],
                    'course_offering' => $video->courseOffering ? [
                        'id' => $video->courseOffering->id,
                        'name' => $video->courseOffering->name,
                    ] : null,
                    'created_at' => $video->created_at,
                ];
            });
            
            return $this->paginated($data, $videos);
        } catch (\Exception $e) {
            Log::error('Instructor Videos API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching videos.', null, 500);
        }
    }
    
    /**
     * Get video details.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $instructor = Auth::user();
            
            $video = ClassSessionVideo::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->with(['course', 'courseOffering'])
                ->firstOrFail();
            
            return $this->success([
                'id' => $video->id,
                'title' => $video->title,
                'description' => $video->description,
                'video_url' => $video->video_url,
                'session_date' => $video->session_date,
                'is_published' => $video->is_published,
                'course' => [
                    'id' => $video->course->id ?? null,
                    'name' => $video->course->name ?? 'N/A',
                ],
                'course_offering' => $video->courseOffering ? [
                    'id' => $video->courseOffering->id,
                    'name' => $video->courseOffering->name,
                ] : null,
                'created_at' => $video->created_at,
                'updated_at' => $video->updated_at,
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Video not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Video API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching video details.', null, 500);
        }
    }
    
    /**
     * Create video.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $instructor = Auth::user();
            
            $validated = $request->validate([
                'course_id' => 'required|exists:courses,id',
                'course_offering_id' => 'nullable|exists:course_offerings,id',
                'title' => 'required|string|max:255',
                'description' => 'nullable|string',
                'video_url' => 'required|url', // This maps to mega_storage_link
                'download_key' => 'nullable|string',
                'thumbnail_url' => 'nullable|url',
                'session_date' => 'required|date',
                'is_published' => 'boolean',
            ]);
            
            // Verify the course belongs to this instructor
            $course = Course::where('id', $validated['course_id'])
                ->where('instructor_id', $instructor->id)
                ->firstOrFail();
            
            $video = ClassSessionVideo::create([
                'course_id' => $validated['course_id'],
                'instructor_id' => $instructor->id,
                'course_offering_id' => $validated['course_offering_id'] ?? null,
                'title' => $validated['title'],
                'description' => $validated['description'] ?? null,
                'mega_storage_link' => $validated['video_url'],
                'download_key' => $validated['download_key'] ?? null,
                'thumbnail_url' => $validated['thumbnail_url'] ?? null,
                'session_date' => $validated['session_date'],
                'is_published' => $validated['is_published'] ?? false,
            ]);
            
            return $this->success([
                'id' => $video->id,
                'title' => $video->title,
            ], 'Video created successfully', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Course not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Video Create API error: ' . $e->getMessage());
            return $this->error('An error occurred while creating video.', null, 500);
        }
    }
    
    /**
     * Update video.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $instructor = Auth::user();
            
            $video = ClassSessionVideo::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->firstOrFail();
            
            $validated = $request->validate([
                'title' => 'sometimes|required|string|max:255',
                'description' => 'nullable|string',
                'video_url' => 'sometimes|required|url',
                'download_key' => 'nullable|string',
                'thumbnail_url' => 'nullable|url',
                'session_date' => 'sometimes|required|date',
                'is_published' => 'boolean',
            ]);
            
            // Map video_url to mega_storage_link
            if (isset($validated['video_url'])) {
                $validated['mega_storage_link'] = $validated['video_url'];
                unset($validated['video_url']);
            }
            
            $video->update($validated);
            
            return $this->success([
                'id' => $video->id,
                'title' => $video->title,
            ], 'Video updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Video not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Video Update API error: ' . $e->getMessage());
            return $this->error('An error occurred while updating video.', null, 500);
        }
    }
    
    /**
     * Delete video.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $instructor = Auth::user();
            
            $video = ClassSessionVideo::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->firstOrFail();
            
            $video->delete();
            
            return $this->success(null, 'Video deleted successfully');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Video not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Video Delete API error: ' . $e->getMessage());
            return $this->error('An error occurred while deleting video.', null, 500);
        }
    }
}

