<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Http\Controllers\Api\V1\BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\Course;
use App\Models\CourseChapter;

class AssignmentController extends BaseController
{
    /**
     * Get list of assignments for instructor's courses.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $instructor = Auth::user();
            
            // Get instructor's course IDs
            $courseIds = Course::where('instructor_id', $instructor->id)
                ->where('is_active', true)
                ->pluck('id');
            
            $query = Assignment::whereIn('course_id', $courseIds)
                ->with(['course', 'chapter']);
            
            // Filter by course
            if ($request->has('course_id') && $request->course_id) {
                $query->where('course_id', $request->course_id);
            }
            
            // Filter by published status
            if ($request->has('published') && $request->published !== '') {
                $query->where('is_published', $request->published == '1');
            }
            
            $assignments = $query->orderBy('due_date', 'desc')
                ->paginate($request->get('per_page', 20));
            
            $data = $assignments->map(function($assignment) {
                // Get submission stats
                $totalSubmissions = AssignmentSubmission::where('assignment_id', $assignment->id)->count();
                $gradedSubmissions = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('status', 'graded')->count();
                $pendingSubmissions = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('status', 'submitted')->count();
                
                return [
                    'id' => $assignment->id,
                    'title' => $assignment->title,
                    'description' => $assignment->description,
                    'due_date' => $assignment->due_date,
                    'due_time' => $assignment->due_time,
                    'total_points' => $assignment->total_points,
                    'is_published' => $assignment->is_published,
                    'course' => [
                        'id' => $assignment->course->id ?? null,
                        'name' => $assignment->course->name ?? 'N/A',
                    ],
                    'chapter' => $assignment->chapter ? [
                        'id' => $assignment->chapter->id,
                        'title' => $assignment->chapter->title,
                    ] : null,
                    'submissions' => [
                        'total' => $totalSubmissions,
                        'graded' => $gradedSubmissions,
                        'pending' => $pendingSubmissions,
                    ],
                    'created_at' => $assignment->created_at,
                ];
            });
            
            return $this->paginated($data, $assignments);
        } catch (\Exception $e) {
            Log::error('Instructor Assignments API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching assignments.', null, 500);
        }
    }
    
    /**
     * Get assignment details.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $instructor = Auth::user();
            
            $assignment = Assignment::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->with(['course', 'chapter', 'submissions.student'])
                ->firstOrFail();
            
            // Get submission statistics
            $totalSubmissions = $assignment->submissions->count();
            $gradedSubmissions = $assignment->submissions->where('status', 'graded')->count();
            $pendingSubmissions = $assignment->submissions->where('status', 'submitted')->count();
            
            // Get enrolled students count
            $enrolledStudents = $assignment->course->enrollments->count();
            
            return $this->success([
                'id' => $assignment->id,
                'title' => $assignment->title,
                'description' => $assignment->description,
                'instructions' => $assignment->instructions,
                'due_date' => $assignment->due_date,
                'due_time' => $assignment->due_time,
                'total_points' => $assignment->total_points,
                'allow_late_submission' => $assignment->allow_late_submission,
                'late_penalty_percentage' => $assignment->late_penalty_percentage,
                'is_published' => $assignment->is_published,
                'course' => [
                    'id' => $assignment->course->id ?? null,
                    'name' => $assignment->course->name ?? 'N/A',
                ],
                'chapter' => $assignment->chapter ? [
                    'id' => $assignment->chapter->id,
                    'title' => $assignment->chapter->title,
                ] : null,
                'statistics' => [
                    'enrolled_students' => $enrolledStudents,
                    'total_submissions' => $totalSubmissions,
                    'graded_submissions' => $gradedSubmissions,
                    'pending_submissions' => $pendingSubmissions,
                ],
                'created_at' => $assignment->created_at,
                'updated_at' => $assignment->updated_at,
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Assignment not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Assignment API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching assignment details.', null, 500);
        }
    }
    
    /**
     * Create assignment.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $instructor = Auth::user();
            
            $validated = $request->validate([
                'course_id' => 'required|exists:courses,id',
                'chapter_id' => 'nullable|exists:course_chapters,id',
                'title' => 'required|string|max:255',
                'description' => 'nullable|string',
                'instructions' => 'nullable|string',
                'due_date' => 'required|date|after_or_equal:today',
                'due_time' => 'nullable|date_format:H:i',
                'total_points' => 'required|numeric|min:0|max:1000',
                'allow_late_submission' => 'boolean',
                'late_penalty_percentage' => 'nullable|integer|min:0|max:100',
                'is_published' => 'boolean',
            ]);
            
            // Verify the course belongs to this instructor
            $course = Course::where('id', $validated['course_id'])
                ->where('instructor_id', $instructor->id)
                ->firstOrFail();
            
            // Verify chapter belongs to the course if provided
            if (isset($validated['chapter_id'])) {
                $chapter = CourseChapter::where('id', $validated['chapter_id'])
                    ->where('course_id', $validated['course_id'])
                    ->firstOrFail();
            }
            
            $assignment = Assignment::create([
                'course_id' => $validated['course_id'],
                'chapter_id' => $validated['chapter_id'] ?? null,
                'title' => $validated['title'],
                'description' => $validated['description'] ?? null,
                'instructions' => $validated['instructions'] ?? null,
                'due_date' => $validated['due_date'],
                'due_time' => $validated['due_time'] ?? null,
                'total_points' => $validated['total_points'],
                'allow_late_submission' => $validated['allow_late_submission'] ?? false,
                'late_penalty_percentage' => $validated['late_penalty_percentage'] ?? 10,
                'is_published' => $validated['is_published'] ?? false,
            ]);
            
            return $this->success([
                'id' => $assignment->id,
                'title' => $assignment->title,
            ], 'Assignment created successfully', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Course or chapter not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Assignment Create API error: ' . $e->getMessage());
            return $this->error('An error occurred while creating assignment.', null, 500);
        }
    }
    
    /**
     * Update assignment.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $instructor = Auth::user();
            
            $assignment = Assignment::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->firstOrFail();
            
            $validated = $request->validate([
                'title' => 'sometimes|required|string|max:255',
                'description' => 'nullable|string',
                'instructions' => 'nullable|string',
                'due_date' => 'sometimes|required|date',
                'due_time' => 'nullable|date_format:H:i',
                'total_points' => 'sometimes|required|numeric|min:0|max:1000',
                'allow_late_submission' => 'boolean',
                'late_penalty_percentage' => 'nullable|integer|min:0|max:100',
                'is_published' => 'boolean',
            ]);
            
            $assignment->update($validated);
            
            return $this->success([
                'id' => $assignment->id,
                'title' => $assignment->title,
            ], 'Assignment updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Assignment not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Assignment Update API error: ' . $e->getMessage());
            return $this->error('An error occurred while updating assignment.', null, 500);
        }
    }
    
    /**
     * Delete assignment.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $instructor = Auth::user();
            
            $assignment = Assignment::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->firstOrFail();
            
            $assignment->delete();
            
            return $this->success(null, 'Assignment deleted successfully');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Assignment not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Assignment Delete API error: ' . $e->getMessage());
            return $this->error('An error occurred while deleting assignment.', null, 500);
        }
    }
    
    /**
     * Get submissions for assignment.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function submissions(Request $request, $id)
    {
        try {
            $instructor = Auth::user();
            
            $assignment = Assignment::where('id', $id)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->firstOrFail();
            
            $query = AssignmentSubmission::where('assignment_id', $id)
                ->with(['student', 'gradedBy']);
            
            // Filter by status
            if ($request->has('status') && $request->status) {
                $query->where('status', $request->status);
            }
            
            $submissions = $query->orderBy('submitted_at', 'desc')
                ->paginate($request->get('per_page', 20));
            
            $data = $submissions->map(function($submission) {
                return [
                    'id' => $submission->id,
                    'student' => [
                        'id' => $submission->student->id ?? null,
                        'name' => $submission->student->name ?? 'N/A',
                        'email' => $submission->student->email ?? 'N/A',
                    ],
                    'submission_text' => $submission->submission_text,
                    'submission_file' => $submission->submission_file ? url('storage/' . $submission->submission_file) : null,
                    'submitted_at' => $submission->submitted_at,
                    'is_late' => $submission->is_late,
                    'status' => $submission->status,
                    'score' => $submission->score,
                    'grade_percentage' => $submission->grade_percentage,
                    'feedback' => $submission->instructor_feedback,
                    'graded_at' => $submission->graded_at,
                    'graded_by' => $submission->gradedBy ? [
                        'id' => $submission->gradedBy->id,
                        'name' => $submission->gradedBy->name,
                    ] : null,
                ];
            });
            
            return $this->paginated($data, $submissions);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Assignment not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Assignment Submissions API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching submissions.', null, 500);
        }
    }
    
    /**
     * Grade a submission.
     *
     * @param Request $request
     * @param int $assignmentId
     * @param int $submissionId
     * @return \Illuminate\Http\JsonResponse
     */
    public function grade(Request $request, $assignmentId, $submissionId)
    {
        try {
            $instructor = Auth::user();
            
            // Verify assignment belongs to instructor
            $assignment = Assignment::where('id', $assignmentId)
                ->whereHas('course', function($q) use ($instructor) {
                    $q->where('instructor_id', $instructor->id);
                })
                ->firstOrFail();
            
            $submission = AssignmentSubmission::where('id', $submissionId)
                ->where('assignment_id', $assignmentId)
                ->firstOrFail();
            
            $validated = $request->validate([
                'score' => 'required|numeric|min:0|max:' . $assignment->total_points,
                'instructor_feedback' => 'nullable|string',
            ], [
                'score.required' => 'Score is required.',
                'score.numeric' => 'Score must be a number.',
                'score.min' => 'Score cannot be negative.',
                'score.max' => 'Score cannot exceed ' . number_format($assignment->total_points, 2) . ' points.',
                'instructor_feedback.string' => 'Feedback must be text.',
            ]);
            
            // Calculate grade percentage and round to 2 decimal places
            $gradePercentage = ($validated['score'] / $assignment->total_points) * 100;
            $gradePercentage = round($gradePercentage, 2);
            
            $submission->score = $validated['score'];
            $submission->grade_percentage = $gradePercentage;
            $submission->instructor_feedback = $validated['instructor_feedback'] ?? null;
            $submission->status = 'graded';
            $submission->graded_at = now();
            $submission->graded_by = $instructor->id;
            $submission->save();
            
            return $this->success([
                'id' => $submission->id,
                'score' => $submission->score,
                'grade_percentage' => $submission->grade_percentage,
                'feedback' => $submission->instructor_feedback,
            ], 'Submission graded successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Assignment or submission not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Instructor Assignment Grade API error: ' . $e->getMessage());
            return $this->error('An error occurred while grading submission.', null, 500);
        }
    }
}

