<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Api\V1\BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\QueryException;
use App\Models\FeePayment;
use App\Models\CourseOffering;

class FeePaymentController extends BaseController
{
    /**
     * Get list of fee payments.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $query = FeePayment::with(['student', 'course', 'courseOffering', 'recordedBy']);
            
            // Search
            if ($request->has('search') && $request->search) {
                $search = $request->search;
                $query->whereHas('student', function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('school_id', 'like', "%{$search}%");
                });
            }
            
            // Filter by date range
            if ($request->has('date_from') && $request->date_from) {
                $query->where('payment_date', '>=', $request->date_from);
            }
            
            if ($request->has('date_to') && $request->date_to) {
                $query->where('payment_date', '<=', $request->date_to);
            }
            
            // Filter by payment method
            if ($request->has('payment_method') && $request->payment_method) {
                $query->where('payment_method', $request->payment_method);
            }
            
            // Filter by payment status
            if ($request->has('payment_status') && $request->payment_status) {
                $query->where('payment_status', $request->payment_status);
            }
            
            $payments = $query->orderBy('payment_date', 'desc')
                ->paginate($request->get('per_page', 50));
            
            $data = $payments->map(function($payment) {
                return [
                    'id' => $payment->id,
                    'amount' => $payment->amount,
                    'payment_date' => $payment->payment_date,
                    'payment_method' => $payment->payment_method,
                    'payment_status' => $payment->payment_status,
                    'transaction_reference' => $payment->transaction_reference,
                    'notes' => $payment->notes,
                    'student' => [
                        'id' => $payment->student->id ?? null,
                        'name' => $payment->student->name ?? 'N/A',
                        'school_id' => $payment->student->school_id ?? null,
                    ],
                    'course' => $payment->course ? [
                        'id' => $payment->course->id,
                        'name' => $payment->course->name,
                    ] : null,
                    'course_offering' => $payment->courseOffering ? [
                        'id' => $payment->courseOffering->id,
                        'name' => $payment->courseOffering->name,
                    ] : null,
                    'recorded_by' => $payment->recordedBy ? [
                        'id' => $payment->recordedBy->id,
                        'name' => $payment->recordedBy->name,
                    ] : null,
                    'created_at' => $payment->created_at,
                ];
            });
            
            $totalAmount = $payments->sum('amount');
            $actualMoneyAtHand = $payments->where('payment_status', 'regular')->sum('amount');
            
            return $this->paginated($data, $payments, [
                'summary' => [
                    'total_amount' => round($totalAmount, 2),
                    'actual_money_at_hand' => round($actualMoneyAtHand, 2),
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Admin Fee Payments API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching fee payments.', null, 500);
        }
    }
    
    /**
     * Create fee payment.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'student_id' => 'required|exists:users,id',
                'course_id' => 'nullable|exists:courses,id',
                'course_offering_id' => 'nullable|exists:course_offerings,id',
                'amount' => 'required|numeric|min:0.01',
                'payment_date' => 'required|date',
                'payment_method' => 'required|in:cash,bank_transfer,card,online',
                'payment_status' => 'required|in:regular,scholarship,refunded',
                'transaction_reference' => 'nullable|string|max:255',
                'notes' => 'nullable|string',
            ]);
            
            // Get course_id - prioritize form value, otherwise get from offering
            $courseId = $validated['course_id'] ?? null;
            if (!$courseId && $validated['course_offering_id']) {
                $offering = CourseOffering::find($validated['course_offering_id']);
                $courseId = $offering ? $offering->course_id : null;
            }
            
            $payment = FeePayment::create([
                'student_id' => $validated['student_id'],
                'course_id' => $courseId,
                'course_offering_id' => $validated['course_offering_id'] ?? null,
                'amount' => $validated['amount'],
                'payment_date' => $validated['payment_date'],
                'payment_method' => $validated['payment_method'],
                'payment_status' => $validated['payment_status'],
                'transaction_reference' => $validated['transaction_reference'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'recorded_by' => Auth::id(),
            ]);
            
            return $this->success([
                'id' => $payment->id,
                'amount' => $payment->amount,
            ], 'Fee payment recorded successfully', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (\Exception $e) {
            Log::error('Admin Fee Payment Create API error: ' . $e->getMessage());
            return $this->error('An error occurred while recording fee payment.', null, 500);
        }
    }
    
    /**
     * Update fee payment.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $payment = FeePayment::findOrFail($id);
            
            $validated = $request->validate([
                'student_id' => 'required|exists:users,id',
                'course_id' => 'nullable|exists:courses,id',
                'course_offering_id' => 'nullable|exists:course_offerings,id',
                'amount' => 'required|numeric|min:0.01',
                'payment_date' => 'required|date',
                'payment_method' => 'required|in:cash,bank_transfer,card,online',
                'payment_status' => 'required|in:regular,scholarship,refunded',
                'transaction_reference' => 'nullable|string|max:255',
                'notes' => 'nullable|string',
            ]);
            
            // Get course_id
            $courseId = $validated['course_id'] ?? null;
            if (!$courseId && $validated['course_offering_id']) {
                $offering = CourseOffering::find($validated['course_offering_id']);
                $courseId = $offering ? $offering->course_id : null;
            }
            
            // Assign validated data
            $payment->student_id = $validated['student_id'];
            $payment->course_id = $courseId;
            $payment->course_offering_id = $validated['course_offering_id'] ?? null;
            $payment->amount = $validated['amount'];
            $payment->payment_date = $validated['payment_date'];
            $payment->payment_method = $validated['payment_method'];
            $payment->payment_status = $validated['payment_status'];
            $payment->transaction_reference = $validated['transaction_reference'] ?? null;
            $payment->notes = $validated['notes'] ?? null;
            
            $payment->save();
            
            return $this->success([
                'id' => $payment->id,
                'amount' => $payment->amount,
            ], 'Fee payment updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::warning('Fee payment validation error: ' . $e->getMessage());
            return $this->error('Validation failed', $e->errors(), 422);
        } catch (QueryException $e) {
            Log::error('Fee payment database update error: ' . $e->getMessage());
            return $this->error('Database error occurred.', ['error' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            Log::error('Fee payment update general error: ' . $e->getMessage());
            return $this->error('An error occurred while updating fee payment.', ['error' => 'An unexpected error occurred.'], 500);
        }
    }
    
    /**
     * Delete fee payment.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $payment = FeePayment::findOrFail($id);
            $payment->delete();
            
            return $this->success(null, 'Fee payment deleted successfully');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->error('Fee payment not found.', null, 404);
        } catch (\Exception $e) {
            Log::error('Admin Fee Payment Delete API error: ' . $e->getMessage());
            return $this->error('An error occurred while deleting fee payment.', null, 500);
        }
    }
}

