<?php

namespace App\Http\Controllers\Api\V1\Admin;

use App\Http\Controllers\Api\V1\BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Models\Course;
use App\Models\Enrollment;
use App\Models\FeePayment;
use App\Models\ClassSessionVideo;
use App\Models\Assignment;
use App\Models\Quiz;

class DashboardController extends BaseController
{
    /**
     * Get admin dashboard statistics.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            // Basic statistics
            $totalStudents = User::where('role', 'student')->count();
            $totalInstructors = User::where('role', 'instructor')->count();
            $totalAdmins = User::where('role', 'admin')->count();
            $totalCourses = Course::where('is_active', true)->count();
            $totalEnrollments = Enrollment::count();
            
            // Financial statistics
            $totalRevenue = FeePayment::sum('amount');
            
            // Calculate pending payments (simplified - count enrollments with outstanding balances)
            $pendingPayments = 0;
            $enrollments = Enrollment::with(['course'])->get();
            foreach ($enrollments as $enrollment) {
                if ($enrollment->course && ($enrollment->course->fees ?? 0) > 0) {
                    $paidAmount = FeePayment::where('student_id', $enrollment->student_id)
                        ->where(function($q) use ($enrollment) {
                            $q->where('course_id', $enrollment->course_id);
                            if ($enrollment->course_offering_id) {
                                $q->orWhere('course_offering_id', $enrollment->course_offering_id);
                            }
                        })
                        ->sum('amount');
                    
                    if ($paidAmount < ($enrollment->course->fees ?? 0)) {
                        $pendingPayments++;
                    }
                }
            }
            
            // Content statistics
            $totalVideos = ClassSessionVideo::where('is_published', true)->count();
            $totalAssignments = Assignment::count();
            $totalQuizzes = Quiz::where('is_active', true)->count();
            
            // Recent enrollments
            $recentEnrollments = Enrollment::with(['student', 'course'])
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get()
                ->map(function($enrollment) {
                    return [
                        'id' => $enrollment->id,
                        'student' => [
                            'id' => $enrollment->student->id ?? null,
                            'name' => $enrollment->student->name ?? 'N/A',
                        ],
                        'course' => [
                            'id' => $enrollment->course->id ?? null,
                            'name' => $enrollment->course->name ?? 'N/A',
                        ],
                        'enrolled_at' => $enrollment->created_at,
                    ];
                });
            
            // Recent fee payments
            $recentPayments = FeePayment::with(['student', 'course'])
                ->orderBy('payment_date', 'desc')
                ->limit(5)
                ->get()
                ->map(function($payment) {
                    return [
                        'id' => $payment->id,
                        'amount' => $payment->amount,
                        'student' => [
                            'id' => $payment->student->id ?? null,
                            'name' => $payment->student->name ?? 'N/A',
                        ],
                        'course' => $payment->course ? [
                            'id' => $payment->course->id,
                            'name' => $payment->course->name,
                        ] : null,
                        'payment_date' => $payment->payment_date,
                    ];
                });
            
            return $this->success([
                'statistics' => [
                    'users' => [
                        'total_students' => $totalStudents,
                        'total_instructors' => $totalInstructors,
                        'total_admins' => $totalAdmins,
                    ],
                    'courses' => [
                        'total_courses' => $totalCourses,
                        'total_enrollments' => $totalEnrollments,
                    ],
                    'financial' => [
                        'total_revenue' => round($totalRevenue, 2),
                        'pending_payments' => round($pendingPayments, 2),
                    ],
                    'content' => [
                        'total_videos' => $totalVideos,
                        'total_assignments' => $totalAssignments,
                        'total_quizzes' => $totalQuizzes,
                    ],
                ],
                'recent_activities' => [
                    'enrollments' => $recentEnrollments,
                    'payments' => $recentPayments,
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Admin Dashboard API error: ' . $e->getMessage());
            return $this->error('An error occurred while fetching dashboard data.', null, 500);
        }
    }
}

