<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Course;
use App\Models\Enrollment;
use App\Models\FeePayment;
use App\Models\CourseOffering;
use App\Models\ClassSessionVideo;
use App\Models\Certificate;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Display the admin dashboard.
     *
     * @return \Illuminate\View\View
     */
    public function dashboard()
    {
        // Get basic statistics
        $totalStudents = User::where('role', 'student')->count();
        $totalInstructors = User::where('role', 'instructor')->count();
        $totalCourses = Course::where('is_active', true)->count();
        $totalEnrollments = Enrollment::count();
        
        // Calculate total revenue (all paid fees)
        $totalRevenue = FeePayment::sum('amount');
        
        // Calculate pending payments (enrollments with outstanding balances)
        $pendingPayments = 0;
        $enrollments = Enrollment::with(['student', 'course'])->get();
        foreach ($enrollments as $enrollment) {
            if ($enrollment->course && $enrollment->course->fees > 0) {
                $paidAmount = FeePayment::where('student_id', $enrollment->student_id)
                    ->where(function($q) use ($enrollment) {
                        $q->where('course_id', $enrollment->course_id);
                        if ($enrollment->course_offering_id) {
                            $q->orWhere('course_offering_id', $enrollment->course_offering_id);
                        }
                    })
                    ->sum('amount');
                
                if ($paidAmount < $enrollment->course->fees) {
                    $pendingPayments++;
                }
            }
        }
        
        // Get recent activities - only get records that exist and have valid relationships
        $recentEnrollments = Enrollment::with(['student', 'course'])
            ->whereHas('student')
            ->whereHas('course')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        $recentPayments = FeePayment::with(['student', 'recordedBy'])
            ->whereHas('student')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        $recentCourses = Course::where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        $activeOfferings = CourseOffering::where('status', 'active')
            ->with('course')
            ->whereHas('course')
            ->orderBy('start_date', 'desc')
            ->limit(5)
            ->get();

        // Calculate additional stats
        $activeCourses = Course::where('is_active', true)->count();
        $activeOfferingsCount = CourseOffering::where('status', 'active')->count();
        $totalCertificates = Certificate::where('is_active', true)->count();
        $totalVideos = ClassSessionVideo::where('is_published', true)->count();
        
        // Get statistics
        $stats = [
            'total_students' => $totalStudents,
            'total_instructors' => $totalInstructors,
            'total_courses' => $totalCourses,
            'total_enrollments' => $totalEnrollments,
            'total_revenue' => $totalRevenue,
            'pending_payments' => $pendingPayments,
            'active_courses' => $activeCourses,
            'active_offerings' => $activeOfferingsCount,
            'total_certificates' => $totalCertificates,
            'total_videos' => $totalVideos,
        ];

        return view('admin.dashboard', compact('stats', 'recentEnrollments', 'recentPayments', 'recentCourses', 'activeOfferings'));
    }
}

