<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Setting;
use Illuminate\Support\Facades\Artisan;

class SettingsController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Display the settings page.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $generalSettings = Setting::getByGroup('general');
        $emailSettings = Setting::getByGroup('email');
        
        return view('admin.settings.index', compact('generalSettings', 'emailSettings'));
    }

    /**
     * Update settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request)
    {
        $group = $request->input('group');
        
        if ($group === 'general') {
            return $this->updateGeneralSettings($request);
        } elseif ($group === 'email') {
            return $this->updateEmailSettings($request);
        } elseif ($group === 'payment') {
            return $this->updatePaymentSettings($request);
        }
        
        return back()->withErrors(['group' => 'Invalid settings group.'])->withInput();
    }

    /**
     * Update general settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    private function updateGeneralSettings(Request $request)
    {
        $validated = $request->validate([
            'system_name' => 'required|string|max:255',
            'system_description' => 'nullable|string|max:500',
            'system_email' => 'required|email|max:255',
            'system_phone' => 'nullable|string|max:20',
            'system_address' => 'nullable|string|max:500',
        ]);
        
        foreach ($validated as $key => $value) {
            Setting::set($key, $value);
        }
        
        Setting::clearCache();
        
        return redirect()->route('admin.settings.index')
            ->with('success', 'General settings updated successfully!');
    }

    /**
     * Update email settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    private function updateEmailSettings(Request $request)
    {
        $validated = $request->validate([
            'mail_mailer' => 'required|string|in:smtp,mailgun,ses,postmark,sendmail',
            'mail_host' => 'required|string|max:255',
            'mail_port' => 'required|string|max:10',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string',
            'mail_encryption' => 'nullable|string|in:tls,ssl,null',
            'mail_from_address' => 'required|email|max:255',
            'mail_from_name' => 'required|string|max:255',
        ]);
        
        // Handle password encryption
        if ($request->has('mail_password') && !empty($request->mail_password)) {
            Setting::setEncrypted('mail_password', $request->mail_password);
        }
        
        // Update other settings
        foreach ($validated as $key => $value) {
            if ($key !== 'mail_password') {
                Setting::set($key, $value);
            }
        }
        
        Setting::clearCache();
        
        // Test email configuration (optional)
        if ($request->has('test_email') && $request->test_email) {
            return $this->testEmail($request);
        }
        
        return redirect()->route('admin.settings.index')
            ->with('success', 'Email settings updated successfully!');
    }

    /**
     * Test email configuration.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function testEmail(Request $request)
    {
        $request->validate([
            'test_email_address' => 'required|email',
        ]);
        
        try {
            // Update email config dynamically
            config([
                'mail.default' => Setting::get('mail_mailer', 'smtp'),
                'mail.mailers.smtp.host' => Setting::get('mail_host'),
                'mail.mailers.smtp.port' => Setting::get('mail_port'),
                'mail.mailers.smtp.username' => Setting::get('mail_username'),
                'mail.mailers.smtp.password' => Setting::getEncrypted('mail_password'),
                'mail.mailers.smtp.encryption' => Setting::get('mail_encryption') === 'null' ? null : Setting::get('mail_encryption'),
                'mail.from.address' => Setting::get('mail_from_address'),
                'mail.from.name' => Setting::get('mail_from_name'),
            ]);
            
            \Mail::raw('This is a test email from ' . Setting::get('system_name') . '. Your email configuration is working correctly!', function ($message) use ($request) {
                $message->to($request->test_email_address)
                        ->subject('Test Email - ' . Setting::get('system_name'));
            });
            
            if ($request->expectsJson()) {
                return response()->json(['success' => true, 'message' => 'Test email sent successfully!']);
            }
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Test email sent successfully to ' . $request->test_email_address . '!');
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json(['success' => false, 'message' => 'Failed to send test email: ' . $e->getMessage()], 500);
            }
            
            return redirect()->route('admin.settings.index')
                ->withErrors(['email_test' => 'Failed to send test email: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Update payment settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    private function updatePaymentSettings(Request $request)
    {
        $validated = $request->validate([
            'payment_info' => 'nullable|string|max:2000',
        ]);
        
        Setting::set('payment_info', $validated['payment_info'] ?? '');
        Setting::clearCache();
        
        return redirect()->route('admin.settings.index')
            ->with('success', 'Payment settings updated successfully!');
    }
}

