<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Quiz;
use App\Models\QuizQuestion;
use App\Models\QuizAnswer;
use App\Models\Course;
use App\Models\Book;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class QuizController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin,instructor']);
    }

    /**
     * Display a listing of quizzes.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = Quiz::with(['course', 'book', 'creator', 'questions']);

        // Instructors can only see their own quizzes
        if (Auth::user()->role === 'instructor') {
            $query->where('created_by', Auth::id());
        }

        // Filter by level
        if ($request->has('level') && $request->level) {
            $query->where('level', $request->level);
        }

        // Filter by course
        if ($request->has('course_id') && $request->course_id) {
            $query->where('course_id', $request->course_id);
        }

        // Filter by book
        if ($request->has('book_id') && $request->book_id) {
            $query->where('book_id', $request->book_id);
        }

        $quizzes = $query->orderBy('level')->orderBy('title')->paginate(20);

        $courses = Course::where('is_active', true)->orderBy('name')->get();
        $books = Book::where('is_active', true)->orderBy('title')->get();

        return view('admin.quizzes.index', compact('quizzes', 'courses', 'books'));
    }

    /**
     * Show the form for creating a new quiz.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $courses = Course::where('is_active', true)->orderBy('name')->get();
        $books = Book::where('is_active', true)->orderBy('title')->get();

        // Instructors can only see their own courses
        if (Auth::user()->role === 'instructor') {
            $courses = $courses->where('instructor_id', Auth::id());
        }

        return view('admin.quizzes.create', compact('courses', 'books'));
    }

    /**
     * Store a newly created quiz.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'course_id' => 'nullable|exists:courses,id',
            'book_id' => 'nullable|exists:books,id',
            'level' => 'required|integer|in:1,2,3',
            'is_active' => 'boolean',
        ]);

        $validated['created_by'] = Auth::id();
        $validated['is_active'] = $request->has('is_active') ? true : false;

        $quiz = Quiz::create($validated);

        return redirect()->route('admin.quizzes.edit', $quiz->id)
            ->with('success', 'Quiz created successfully! Now add questions.');
    }

    /**
     * Show the form for editing the specified quiz.
     *
     * @param  Quiz  $quiz
     * @return \Illuminate\View\View
     */
    public function edit(Quiz $quiz)
    {
        // Instructors can only edit their own quizzes
        if (Auth::user()->role === 'instructor' && $quiz->created_by !== Auth::id()) {
            abort(403);
        }

        $quiz->load(['questions.answers']);
        $courses = Course::where('is_active', true)->orderBy('name')->get();
        $books = Book::where('is_active', true)->orderBy('title')->get();

        if (Auth::user()->role === 'instructor') {
            $courses = $courses->where('instructor_id', Auth::id());
        }

        return view('admin.quizzes.edit', compact('quiz', 'courses', 'books'));
    }

    /**
     * Update the specified quiz.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Quiz  $quiz
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Quiz $quiz)
    {
        // Instructors can only update their own quizzes
        if (Auth::user()->role === 'instructor' && $quiz->created_by !== Auth::id()) {
            abort(403);
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'course_id' => 'nullable|exists:courses,id',
            'book_id' => 'nullable|exists:books,id',
            'level' => 'required|integer|in:1,2,3',
            'time_limit_minutes' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->has('is_active') ? true : false;

        $quiz->update($validated);

        return redirect()->route('admin.quizzes.edit', $quiz->id)
            ->with('success', 'Quiz updated successfully!');
    }

    /**
     * Store a question for the quiz.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Quiz  $quiz
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storeQuestion(Request $request, Quiz $quiz)
    {
        $validated = $request->validate([
            'question' => 'required|string',
            'question_type' => 'required|in:multiple_choice,true_false',
            'explanation' => 'nullable|string',
            'answers' => 'required|array|min:2',
            'answers.*.text' => 'required|string',
            'answers.*.is_correct' => 'nullable',
            'correct_answer' => 'nullable|integer',
        ]);

        DB::transaction(function () use ($quiz, $validated, $request) {
            $question = QuizQuestion::create([
                'quiz_id' => $quiz->id,
                'question' => $validated['question'],
                'question_type' => $validated['question_type'],
                'explanation' => $validated['explanation'] ?? null,
                'order' => $quiz->questions()->count() + 1,
            ]);

            $correctAnswerIndex = $request->input('correct_answer', 0);
            
            foreach ($validated['answers'] as $index => $answerData) {
                // Check if is_correct is set directly, or use correct_answer index
                $isCorrect = false;
                if (isset($answerData['is_correct'])) {
                    $isCorrect = $answerData['is_correct'] == '1' || $answerData['is_correct'] === true || $answerData['is_correct'] === 'true';
                } else {
                    $isCorrect = $index == $correctAnswerIndex;
                }
                
                QuizAnswer::create([
                    'question_id' => $question->id,
                    'answer_text' => $answerData['text'],
                    'is_correct' => $isCorrect,
                    'order' => $index + 1,
                ]);
            }
        });

        return redirect()->route('admin.quizzes.edit', $quiz->id)
            ->with('success', 'Question added successfully!');
    }

    /**
     * Update a question.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Quiz  $quiz
     * @param  QuizQuestion  $question
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateQuestion(Request $request, Quiz $quiz, QuizQuestion $question)
    {
        if ($question->quiz_id !== $quiz->id) {
            abort(404);
        }

        $validated = $request->validate([
            'question' => 'required|string',
            'question_type' => 'required|in:multiple_choice,true_false',
            'explanation' => 'nullable|string',
            'answers' => 'required|array|min:2',
            'answers.*.id' => 'nullable|exists:quiz_answers,id',
            'answers.*.text' => 'required|string',
            'answers.*.is_correct' => 'required|boolean',
        ]);

        DB::transaction(function () use ($question, $validated) {
            $question->update([
                'question' => $validated['question'],
                'question_type' => $validated['question_type'],
                'explanation' => $validated['explanation'] ?? null,
            ]);

            // Get existing answer IDs
            $existingAnswerIds = $question->answers->pluck('id')->toArray();
            $newAnswerIds = [];

            foreach ($validated['answers'] as $index => $answerData) {
                if (isset($answerData['id']) && in_array($answerData['id'], $existingAnswerIds)) {
                    // Update existing answer
                    QuizAnswer::where('id', $answerData['id'])->update([
                        'answer_text' => $answerData['text'],
                        'is_correct' => $answerData['is_correct'],
                        'order' => $index + 1,
                    ]);
                    $newAnswerIds[] = $answerData['id'];
                } else {
                    // Create new answer
                    $newAnswer = QuizAnswer::create([
                        'question_id' => $question->id,
                        'answer_text' => $answerData['text'],
                        'is_correct' => $answerData['is_correct'],
                        'order' => $index + 1,
                    ]);
                    $newAnswerIds[] = $newAnswer->id;
                }
            }

            // Delete answers that were removed
            QuizAnswer::where('question_id', $question->id)
                ->whereNotIn('id', $newAnswerIds)
                ->delete();
        });

        return redirect()->route('admin.quizzes.edit', $quiz->id)
            ->with('success', 'Question updated successfully!');
    }

    /**
     * Delete a question.
     *
     * @param  Quiz  $quiz
     * @param  QuizQuestion  $question
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deleteQuestion(Quiz $quiz, QuizQuestion $question)
    {
        if ($question->quiz_id !== $quiz->id) {
            abort(404);
        }

        $question->delete();

        return redirect()->route('admin.quizzes.edit', $quiz->id)
            ->with('success', 'Question deleted successfully!');
    }

    /**
     * Export quiz as JSON template.
     *
     * @param  Quiz  $quiz
     * @return \Illuminate\Http\Response
     */
    public function export(Quiz $quiz)
    {
        $quiz->load(['questions.answers']);

        $exportData = [
            'title' => $quiz->title,
            'description' => $quiz->description,
            'level' => $quiz->level,
            'course_id' => $quiz->course_id,
            'book_id' => $quiz->book_id,
            'questions' => [],
        ];

        foreach ($quiz->questions as $question) {
            $exportData['questions'][] = [
                'question' => $question->question,
                'question_type' => $question->question_type,
                'explanation' => $question->explanation,
                'answers' => $question->answers->map(function ($answer) {
                    return [
                        'text' => $answer->answer_text,
                        'is_correct' => $answer->is_correct,
                    ];
                })->toArray(),
            ];
        }

        $filename = 'quiz_' . str_replace(' ', '_', $quiz->title) . '_' . date('Y-m-d') . '.json';

        return response()->json($exportData, 200, [
            'Content-Type' => 'application/json',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ], JSON_PRETTY_PRINT);
    }

    /**
     * Show import form.
     *
     * @return \Illuminate\View\View
     */
    public function showImport()
    {
        $courses = Course::where('is_active', true)->orderBy('name')->get();
        $books = Book::where('is_active', true)->orderBy('title')->get();

        if (Auth::user()->role === 'instructor') {
            $courses = $courses->where('instructor_id', Auth::id());
        }

        return view('admin.quizzes.import', compact('courses', 'books'));
    }

    /**
     * Import quiz from JSON.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function import(Request $request)
    {
        $validated = $request->validate([
            'json_file' => 'required|file|mimes:json,txt|max:5120', // Max 5MB
            'course_id' => 'nullable|exists:courses,id',
            'book_id' => 'nullable|exists:books,id',
        ]);

        // Additional security: Check file size
        if ($validated['json_file']->getSize() > 5 * 1024 * 1024) {
            return redirect()->back()->with('error', 'File size exceeds maximum allowed size of 5MB.');
        }

        // Read file safely
        $jsonContent = file_get_contents($validated['json_file']->getRealPath());
        
        // Validate JSON structure before decoding
        if (empty($jsonContent)) {
            return redirect()->back()->with('error', 'File is empty.');
        }
        
        $data = json_decode($jsonContent, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return redirect()->back()->with('error', 'Invalid JSON file: ' . json_last_error_msg());
        }

        // Validate structure
        if (!isset($data['title']) || !isset($data['questions']) || !is_array($data['questions'])) {
            return redirect()->back()->with('error', 'Invalid JSON structure. Please use the template format.');
        }

        DB::transaction(function () use ($data, $validated) {
            $quiz = Quiz::create([
                'title' => $data['title'],
                'description' => $data['description'] ?? null,
                'level' => $data['level'] ?? 1,
                'course_id' => $validated['course_id'] ?? $data['course_id'] ?? null,
                'book_id' => $validated['book_id'] ?? $data['book_id'] ?? null,
                'is_active' => true,
                'created_by' => Auth::id(),
            ]);

            foreach ($data['questions'] as $index => $questionData) {
                $question = QuizQuestion::create([
                    'quiz_id' => $quiz->id,
                    'question' => $questionData['question'],
                    'question_type' => $questionData['question_type'] ?? 'multiple_choice',
                    'explanation' => $questionData['explanation'] ?? null,
                    'order' => $index + 1,
                ]);

                foreach ($questionData['answers'] as $answerIndex => $answerData) {
                    QuizAnswer::create([
                        'question_id' => $question->id,
                        'answer_text' => $answerData['text'],
                        'is_correct' => $answerData['is_correct'] ?? false,
                        'order' => $answerIndex + 1,
                    ]);
                }
            }
        });

        return redirect()->route('admin.quizzes.index')
            ->with('success', 'Quiz imported successfully!');
    }

    /**
     * Download JSON template.
     *
     * @return \Illuminate\Http\Response
     */
    public function downloadTemplate()
    {
        $template = [
            'title' => 'Sample Quiz Title',
            'description' => 'Quiz description (optional)',
            'level' => 1,
            'course_id' => null,
            'book_id' => null,
            'questions' => [
                [
                    'question' => 'What is Flutter?',
                    'question_type' => 'multiple_choice',
                    'explanation' => 'Flutter is Google\'s UI toolkit for building natively compiled applications.',
                    'answers' => [
                        ['text' => 'A programming language', 'is_correct' => false],
                        ['text' => 'A UI toolkit for building mobile apps', 'is_correct' => true],
                        ['text' => 'A database system', 'is_correct' => false],
                        ['text' => 'A web framework', 'is_correct' => false],
                    ],
                ],
                [
                    'question' => 'Dart is the programming language used by Flutter.',
                    'question_type' => 'true_false',
                    'explanation' => 'Yes, Flutter uses Dart as its programming language.',
                    'answers' => [
                        ['text' => 'True', 'is_correct' => true],
                        ['text' => 'False', 'is_correct' => false],
                    ],
                ],
            ],
        ];

        return response()->json($template, 200, [
            'Content-Type' => 'application/json',
            'Content-Disposition' => 'attachment; filename="quiz_template.json"',
        ], JSON_PRETTY_PRINT);
    }

    /**
     * Remove the specified quiz.
     *
     * @param  Quiz  $quiz
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Quiz $quiz)
    {
        // Instructors can only delete their own quizzes
        if (Auth::user()->role === 'instructor' && $quiz->created_by !== Auth::id()) {
            abort(403);
        }

        $quiz->delete();

        return redirect()->route('admin.quizzes.index')
            ->with('success', 'Quiz deleted successfully!');
    }
}

