<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use App\Mail\InstructorWelcomeMail;

class InstructorController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Display a listing of instructors.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = User::where('role', 'instructor');

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $instructors = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.instructors.index', compact('instructors'));
    }

    /**
     * Show the form for creating a new instructor.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $existingCourses = Course::where('is_active', true)->orderBy('name')->get();
        return view('admin.instructors.create', compact('existingCourses'));
    }

    /**
     * Store a newly created instructor.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'password' => 'required|string|min:8|confirmed',
            'courses' => 'nullable',
            'new_courses' => 'nullable|array',
            'new_courses.*.name' => 'nullable|string|max:255',
            'new_courses.*.code' => 'nullable|string|max:50',
            'new_courses.*.description' => 'nullable|string',
            'new_courses.*.credits' => 'nullable|integer|min:0',
            'new_courses.*.fees' => 'nullable|numeric|min:0',
        ]);
        
        // Normalize courses to always be an array
        if (isset($validated['courses'])) {
            if (is_string($validated['courses'])) {
                // If it's a string, convert to array
                $validated['courses'] = [$validated['courses']];
            } elseif (!is_array($validated['courses'])) {
                // If it's not an array, make it empty
                $validated['courses'] = [];
            }
            // Validate that all course IDs exist
            if (!empty($validated['courses'])) {
                $existingCourses = \App\Models\Course::whereIn('id', $validated['courses'])->pluck('id')->toArray();
                $invalidCourses = array_diff($validated['courses'], $existingCourses);
                if (!empty($invalidCourses)) {
                    throw \Illuminate\Validation\ValidationException::withMessages([
                        'courses' => 'One or more selected courses do not exist.',
                    ]);
                }
            }
        } else {
            $validated['courses'] = [];
        }
        
        // Custom validation for new_courses - only validate if the course entry has at least name or code
        if (!empty($validated['new_courses']) && is_array($validated['new_courses'])) {
            foreach ($validated['new_courses'] as $index => $course) {
                // Ensure course is an array
                if (!is_array($course)) {
                    unset($validated['new_courses'][$index]);
                    continue;
                }
                
                // If any field is filled, both name and code are required
                if (!empty($course['name']) || !empty($course['code']) || !empty($course['description'])) {
                    if (empty($course['name'])) {
                        throw \Illuminate\Validation\ValidationException::withMessages([
                            "new_courses.{$index}.name" => 'The course name field is required when creating a new course.',
                        ]);
                    }
                    if (empty($course['code'])) {
                        throw \Illuminate\Validation\ValidationException::withMessages([
                            "new_courses.{$index}.code" => 'The course code field is required when creating a new course.',
                        ]);
                    }
                }
            }
            // Filter out empty course entries and reindex array
            $validated['new_courses'] = array_values(array_filter($validated['new_courses'], function($course) {
                return is_array($course) && !empty($course['name']) && !empty($course['code']);
            }));
        } else {
            $validated['new_courses'] = [];
        }

        $instructor = null;
        $password = $validated['password'];
        
        DB::transaction(function () use ($validated, &$instructor) {
            // Create the instructor user
            $instructor = User::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'phone' => $validated['phone'] ?? null,
                'address' => $validated['address'] ?? null,
                'password' => Hash::make($validated['password']),
                'role' => 'instructor',
                'email_verified_at' => now(),
            ]);

            // Assign existing courses if selected
            if (!empty($validated['courses'])) {
                Course::whereIn('id', $validated['courses'])
                    ->update(['instructor_id' => $instructor->id]);
            }

            // Create and assign new courses
            if (!empty($validated['new_courses'])) {
                foreach ($validated['new_courses'] as $courseData) {
                    if (!empty($courseData['name']) && !empty($courseData['code'])) {
                        // Check if course code already exists
                        $existingCourse = Course::where('code', $courseData['code'])->first();
                        if ($existingCourse) {
                            // If course exists, just assign it to this instructor
                            $existingCourse->update(['instructor_id' => $instructor->id]);
                        } else {
                            // Create new course
                            Course::create([
                                'name' => $courseData['name'],
                                'code' => $courseData['code'],
                                'description' => $courseData['description'] ?? null,
                                'instructor_id' => $instructor->id,
                                'credits' => $courseData['credits'] ?? 0,
                                'fees' => $courseData['fees'] ?? 0.00,
                                'is_active' => true,
                            ]);
                        }
                    }
                }
            }
        });

        // Send welcome email
        try {
            Mail::to($instructor->email)->send(new InstructorWelcomeMail($instructor, $password));
        } catch (\Exception $e) {
            // Log error but don't fail the request
            \Log::error('Failed to send welcome email to instructor: ' . $e->getMessage());
        }

        return redirect()->route('admin.instructors.index')
            ->with('success', 'Instructor created successfully with assigned courses! Welcome email sent.');
    }

    /**
     * Display the specified instructor.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $instructor = User::where('role', 'instructor')->findOrFail($id);
        $courses = Course::where('instructor_id', $instructor->id)->get();
        return view('admin.instructors.show', compact('instructor', 'courses'));
    }

    /**
     * Show the form for editing the specified instructor.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $instructor = User::where('role', 'instructor')->findOrFail($id);
        return view('admin.instructors.edit', compact('instructor'));
    }

    /**
     * Update the specified instructor.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $instructor = User::where('role', 'instructor')->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($instructor->id)],
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        $instructor->name = $validated['name'];
        $instructor->email = $validated['email'];
        $instructor->phone = $validated['phone'] ?? null;
        $instructor->address = $validated['address'] ?? null;

        if (!empty($validated['password'])) {
            $instructor->password = Hash::make($validated['password']);
        }

        $instructor->save();

        return redirect()->route('admin.instructors.index')
            ->with('success', 'Instructor updated successfully!');
    }

    /**
     * Remove the specified instructor.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        $instructor = User::where('role', 'instructor')->findOrFail($id);
        $instructor->delete();

        return redirect()->route('admin.instructors.index')
            ->with('success', 'Instructor deleted successfully!');
    }
}

