<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\User;
use Illuminate\Http\Request;

class CourseController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Display a listing of courses.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = Course::with('instructor');

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Filter by instructor
        if ($request->has('instructor_id') && $request->instructor_id) {
            $query->where('instructor_id', $request->instructor_id);
        }

        // Filter by status
        if ($request->has('status') && $request->status !== '') {
            $query->where('is_active', $request->status == 'active');
        }

        $courses = $query->orderBy('created_at', 'desc')->paginate(15);
        $instructors = User::where('role', 'instructor')->orderBy('name')->get();

        return view('admin.courses.index', compact('courses', 'instructors'));
    }

    /**
     * Show the form for creating a new course.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $instructors = User::where('role', 'instructor')->orderBy('name')->get();
        return view('admin.courses.create', compact('instructors'));
    }

    /**
     * Store a newly created course.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:courses,code',
            'description' => 'nullable|string',
            'instructor_id' => 'required|exists:users,id',
            'credits' => 'nullable|integer|min:0',
            'fees' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
            'academic_session' => 'nullable|string|max:255',
            'from_month' => 'nullable|string|max:100',
            'to_month' => 'nullable|string|max:100',
            'number_of_sessions' => 'nullable|integer|min:0',
            'number_of_weeks' => 'nullable|integer|min:0',
            'schedule_info' => 'nullable|string',
            'schedule_days' => 'nullable|array',
            'schedule_days.*' => 'string|in:Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'session_duration_minutes' => 'nullable|integer|min:1',
        ]);

        // Build session_period from from_month and to_month
        $sessionPeriod = null;
        if (!empty($validated['from_month']) && !empty($validated['to_month'])) {
            $sessionPeriod = $validated['from_month'] . ' - ' . $validated['to_month'];
        }

        Course::create([
            'name' => $validated['name'],
            'code' => $validated['code'],
            'description' => $validated['description'] ?? null,
            'instructor_id' => $validated['instructor_id'],
            'credits' => $validated['credits'] ?? 0,
            'fees' => $validated['fees'] ?? 0.00,
            'is_active' => $validated['is_active'] ?? true,
            'academic_session' => $validated['academic_session'] ?? null,
            'session_period' => $sessionPeriod,
            'from_month' => $validated['from_month'] ?? null,
            'to_month' => $validated['to_month'] ?? null,
            'number_of_sessions' => $validated['number_of_sessions'] ?? 0,
            'number_of_weeks' => $validated['number_of_weeks'] ?? 0,
            'schedule_info' => $validated['schedule_info'] ?? null,
            'schedule_days' => $validated['schedule_days'] ?? null,
            'start_time' => !empty($validated['start_time']) ? $validated['start_time'] : null,
            'end_time' => !empty($validated['end_time']) ? $validated['end_time'] : null,
            'session_duration_minutes' => $validated['session_duration_minutes'] ?? 90,
        ]);

        return redirect()->route('admin.courses.index')
            ->with('success', 'Course created successfully!');
    }

    /**
     * Display the specified course.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $course = Course::with(['instructor', 'chapters'])->findOrFail($id);
        $chapters = $course->chapters()->orderBy('order')->get();
        return view('admin.courses.show', compact('course', 'chapters'));
    }

    /**
     * Show the form for editing the specified course.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $course = Course::findOrFail($id);
        $instructors = User::where('role', 'instructor')->orderBy('name')->get();
        return view('admin.courses.edit', compact('course', 'instructors'));
    }

    /**
     * Update the specified course.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $course = Course::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => ['required', 'string', 'max:50', \Illuminate\Validation\Rule::unique('courses')->ignore($course->id)],
            'description' => 'nullable|string',
            'instructor_id' => 'required|exists:users,id',
            'credits' => 'nullable|integer|min:0',
            'fees' => 'nullable|numeric|min:0',
            'is_active' => 'nullable|boolean',
            'academic_session' => 'nullable|string|max:255',
            'from_month' => 'nullable|string|max:100',
            'to_month' => 'nullable|string|max:100',
            'number_of_sessions' => 'nullable|integer|min:0',
            'number_of_weeks' => 'nullable|integer|min:0',
            'schedule_info' => 'nullable|string',
            'schedule_days' => 'nullable|array',
            'schedule_days.*' => 'string|in:Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'session_duration_minutes' => 'nullable|integer|min:1',
        ]);

        // Build session_period from from_month and to_month
        $sessionPeriod = null;
        if (!empty($validated['from_month']) && !empty($validated['to_month'])) {
            $sessionPeriod = $validated['from_month'] . ' - ' . $validated['to_month'];
        }
        $validated['session_period'] = $sessionPeriod;

        $course->update($validated);

        return redirect()->route('admin.courses.index')
            ->with('success', 'Course updated successfully!');
    }

    /**
     * Remove the specified course.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        $course = Course::findOrFail($id);
        $course->delete();

        return redirect()->route('admin.courses.index')
            ->with('success', 'Course deleted successfully!');
    }
}

