<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseChapter;
use Illuminate\Http\Request;

class CourseChapterController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Show the form for creating a new chapter.
     *
     * @param  int  $courseId
     * @return \Illuminate\View\View
     */
    public function create($courseId)
    {
        $course = Course::findOrFail($courseId);
        $nextChapterNumber = $course->chapters()->max('chapter_number') + 1;
        return view('admin.courses.chapters.create', compact('course', 'nextChapterNumber'));
    }

    /**
     * Store a newly created chapter.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $courseId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request, $courseId)
    {
        $course = Course::findOrFail($courseId);

        $validated = $request->validate([
            'chapter_number' => 'required|integer|min:1',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'session_number' => 'nullable|integer|min:1',
            'order' => 'nullable|integer|min:0',
            'is_published' => 'nullable|boolean',
        ]);

        CourseChapter::create([
            'course_id' => $course->id,
            'chapter_number' => $validated['chapter_number'],
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'content' => $validated['content'] ?? null,
            'session_number' => $validated['session_number'] ?? null,
            'order' => $validated['order'] ?? $validated['chapter_number'],
            'is_published' => $validated['is_published'] ?? false,
        ]);

        return redirect()->route('admin.courses.show', $course->id)
            ->with('success', 'Chapter created successfully!');
    }

    /**
     * Display the specified chapter.
     *
     * @param  int  $courseId
     * @param  int  $chapterId
     * @return \Illuminate\View\View
     */
    public function show($courseId, $chapterId)
    {
        $course = Course::findOrFail($courseId);
        $chapter = CourseChapter::where('course_id', $courseId)->findOrFail($chapterId);
        return view('admin.courses.chapters.show', compact('course', 'chapter'));
    }

    /**
     * Show the form for editing the specified chapter.
     *
     * @param  int  $courseId
     * @param  int  $chapterId
     * @return \Illuminate\View\View
     */
    public function edit($courseId, $chapterId)
    {
        $course = Course::findOrFail($courseId);
        $chapter = CourseChapter::where('course_id', $courseId)->findOrFail($chapterId);
        return view('admin.courses.chapters.edit', compact('course', 'chapter'));
    }

    /**
     * Update the specified chapter.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $courseId
     * @param  int  $chapterId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $courseId, $chapterId)
    {
        $course = Course::findOrFail($courseId);
        $chapter = CourseChapter::where('course_id', $courseId)->findOrFail($chapterId);

        $validated = $request->validate([
            'chapter_number' => 'required|integer|min:1',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'session_number' => 'nullable|integer|min:1',
            'order' => 'nullable|integer|min:0',
            'is_published' => 'nullable|boolean',
        ]);

        $chapter->update($validated);

        return redirect()->route('admin.courses.show', $course->id)
            ->with('success', 'Chapter updated successfully!');
    }

    /**
     * Remove the specified chapter.
     *
     * @param  int  $courseId
     * @param  int  $chapterId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($courseId, $chapterId)
    {
        $course = Course::findOrFail($courseId);
        $chapter = CourseChapter::where('course_id', $courseId)->findOrFail($chapterId);
        $chapter->delete();

        return redirect()->route('admin.courses.show', $course->id)
            ->with('success', 'Chapter deleted successfully!');
    }
}


