<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Models\Notification;
use App\Models\User;
use App\Mail\AnnouncementMail;

class AnnouncementController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Display the announcement page.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $recentAnnouncements = Notification::whereIn('type', ['announcement'])
            ->with(['sender', 'user'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('admin.announcements.index', compact('recentAnnouncements'));
    }

    /**
     * Store a new announcement.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'recipients' => 'required|array|min:1',
            'recipients.*' => 'in:all_students,all_instructors,students,instructors',
            'priority' => 'required|in:low,normal,high,urgent',
            'send_email' => 'boolean',
        ]);

        $sender = Auth::user();
        $recipientIds = [];

        // Determine recipient user IDs
        if (in_array('all_students', $validated['recipients'])) {
            $recipientIds = array_merge($recipientIds, User::where('role', 'student')->pluck('id')->toArray());
        }
        if (in_array('all_instructors', $validated['recipients'])) {
            $recipientIds = array_merge($recipientIds, User::where('role', 'instructor')->pluck('id')->toArray());
        }
        if (in_array('students', $validated['recipients']) && $request->has('student_ids') && is_array($request->student_ids)) {
            $recipientIds = array_merge($recipientIds, $request->student_ids);
        }
        if (in_array('instructors', $validated['recipients']) && $request->has('instructor_ids') && is_array($request->instructor_ids)) {
            $recipientIds = array_merge($recipientIds, $request->instructor_ids);
        }

        // Remove duplicates
        $recipientIds = array_unique($recipientIds);

        if (empty($recipientIds)) {
            return back()->withErrors(['recipients' => 'Please select at least one recipient.'])->withInput();
        }

        // Create notifications for each recipient
        $notifications = [];
        $emailRecipients = [];

        foreach ($recipientIds as $userId) {
            $user = User::find($userId);
            if (!$user) continue;

            $notification = Notification::create([
                'user_id' => $userId,
                'sender_id' => $sender->id,
                'type' => 'announcement',
                'title' => $validated['title'],
                'message' => $validated['message'],
                'priority' => $validated['priority'],
            ]);

            $notifications[] = $notification;

            // Prepare email recipients if email sending is enabled
            if ($request->has('send_email') && $user->email) {
                $emailRecipients[] = [
                    'user' => $user,
                    'notification' => $notification,
                ];
            }
        }

        // Send emails if requested
        if ($request->has('send_email') && !empty($emailRecipients)) {
            foreach ($emailRecipients as $recipient) {
                try {
                    Mail::to($recipient['user']->email)->send(
                        new AnnouncementMail($recipient['notification'])
                    );
                    
                    $recipient['notification']->update([
                        'email_sent' => true,
                        'email_sent_at' => now(),
                    ]);
                } catch (\Exception $e) {
                    // Log error but don't fail the entire operation
                    \Log::error('Failed to send announcement email: ' . $e->getMessage());
                }
            }
        }

        return redirect()->route('admin.announcements.index')
            ->with('success', 'Announcement sent to ' . count($notifications) . ' recipient(s) successfully!');
    }
}

